/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */

package com.queplix.core.modules.services.utils.log;

import com.queplix.core.integrator.security.LogonSession;
import com.queplix.core.utils.DateHelper;

import java.io.CharArrayWriter;
import java.io.PrintWriter;
import java.io.Serializable;

/**
 * Base class for a log record value object.
 *
 * @author [ONZ] Oleg N. Zhovtanyuk
 * @author [ALB] Andrey Baranov
 * @version $Revision: 1.1.1.1 $ $Date: 2005/09/12 15:31:02 $
 */
public abstract class AbstractLogRecord
        implements Serializable {

    // ================================================================== Fields

    // Current logon session
    private LogonSession ls;

    // Log message level.
    private Level level;

    // Log message.
    private String message;

    // Exception cause.
    private Throwable cause;

    // Exception stack trace.
    private String stack;

    // Event system time.
    private long time = DateHelper.currentTimeMillis();

    // ========================================================== Initialization

    /**
     * Creates a new log record.
     *
     * @param ls      user logon session
     * @param level   log level
     * @param message the message to log
     */
    public AbstractLogRecord(LogonSession ls, Level level, String message) {

        // Check arguments for null.
        if(ls == null) {
            throw new NullPointerException("Logon session is null.");
        } else if(level == null) {
            throw new NullPointerException("Log level is null.");
        }

        // Store data.
        this.ls = ls;
        this.level = level;
        this.message = message;

    }

    // ========================================================== Public methods

    /**
     * Log handler class name getter.
     *
     * @return class name
     */
    public abstract String getHndClassName();

    /**
     * User logon session getter.
     *
     * @return user logon session
     */
    public LogonSession getLogonSession() {
        return ls;
    }

    /**
     * Log message level getter.
     *
     * @return log message level
     */
    public Level getLevel() {
        return level;
    }

    /**
     * Log message getter.
     *
     * @return log message
     */
    public String getMessage() {
        return message;
    }

    /**
     * Exception cause getter.
     *
     * @return exception cause
     */
    public Throwable getCause() {
        return cause;
    }

    /**
     * Exception stack trace getter.
     *
     * @return stack trace
     */
    public String getStackTrace() {
        return stack;
    }

    /**
     * Log event time getter.
     *
     * @return log event time
     */
    public long getTime() {
        return time;
    }

    /**
     * Exception cause setter.
     * Sets the exception cause and stack trace.
     *
     * @param cause exception cause
     */
    public void setCause(Throwable cause) {
        this.cause = cause;
        if(cause != null) {
            CharArrayWriter caw = new CharArrayWriter();
            cause.printStackTrace(new PrintWriter(caw));
            stack = caw.toString();
            caw.close();
        }
    }

    // ======================================================= Inherited methods

    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */

    public final String toString() {
        StringBuffer sb = new StringBuffer();
        sb.append("[time = ").append(getTime());
        sb.append(", level = ").append(getLevel());
        sb.append(", message = '").append(getMessage()).append("'");
        sb.append(", stack = '").append(getStackTrace()).append("'");
        sb.append(", handler = ").append(getHndClassName());
        sb.append("]");
        return sb.toString();
    }

}
