/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */
package com.queplix.core.client.common.ui;

import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.DeckPanel;
import com.google.gwt.user.client.ui.HasWidgets;
import com.google.gwt.user.client.ui.IndexedPanel;
import com.google.gwt.user.client.ui.SourcesTabEvents;
import com.google.gwt.user.client.ui.TabListener;
import com.google.gwt.user.client.ui.TabListenerCollection;
import com.google.gwt.user.client.ui.VerticalPanel;
import com.google.gwt.user.client.ui.Widget;
import com.google.gwt.user.client.ui.WidgetCollection;

import java.util.Iterator;

/**
 * Fork of GWT TabPanel which supports QTabBars instead of GWT TabBars
 * All the original code belongs to Google Inc.
 *
 * @see com.google.gwt.user.client.ui.TabPanel
 * @author: Vasily Mikhailitchenko
 * @since: 12 Feb 2007
 */

/**
 * @noinspection ALL
 */
public class QTabPanelBase extends Composite implements TabListener,
        SourcesTabEvents, HasWidgets, IndexedPanel {

    private WidgetCollection children = new WidgetCollection(this);
    private DeckPanel deck = new DeckPanel();
    private QTabBar qTabBar;
    private TabListenerCollection tabListeners;

    /**
     * Creates an empty tab panel.
     */
    public QTabPanelBase() {
        this(new QTabBar());
    }

    public QTabPanelBase(QTabBar qTabBar) {
        setTabBar(qTabBar);
        VerticalPanel panel = new VerticalPanel();
        panel.add(this.qTabBar);
        panel.add(deck);

        panel.setCellHeight(deck, "100%");
        this.qTabBar.setWidth("100%");
        this.qTabBar.addTabListener(this);
        initWidget(panel);
        setStyleName("gwt-TabPanel");
        deck.setStyleName("gwt-TabPanelBottom");
    }

    public void setTabBar(QTabBar tabBar) {
        this.qTabBar = tabBar;
        this.qTabBar.setInTabPanel(true);
    }

    public void add(Widget w) {
        throw new UnsupportedOperationException(
                "A tabText parameter must be specified with add().");
    }

    /**
     * Adds a widget to the tab panel.
     *
     * @param w       the widget to be added
     * @param tabText the text to be shown on its tab
     */
    public void add(Widget w, String tabText) {
        insert(w, tabText, getWidgetCount());
    }

    /**
     * Adds a widget to the tab panel.
     *
     * @param w       the widget to be added
     * @param tabText the text to be shown on its tab
     * @param asHTML  <code>true</code> to treat the specified text as HTML
     */
    public void add(Widget w, String tabText, boolean asHTML) {
        insert(w, tabText, asHTML, getWidgetCount());
    }

    public void addTabListener(TabListener listener) {
        if(tabListeners == null) {
            tabListeners = new TabListenerCollection();
        }
        tabListeners.add(listener);
    }

    public void clear() {
        while(getWidgetCount() > 0) {
            remove(getWidget(0));
        }
    }

    /**
     * Gets the deck panel within this tab panel.
     *
     * @return the deck panel
     */
    public DeckPanel getDeckPanel() {
        return deck;
    }

    /**
     * Gets the tab bar within this tab panel.
     *
     * @return the tab bar
     */
    public QTabBar getTabBar() {
        return qTabBar;
    }

    public Widget getWidget(int index) {
        return children.get(index);
    }

    public int getWidgetCount() {
        return children.size();
    }

    public int getWidgetIndex(Widget widget) {
        return children.indexOf(widget);
    }

    /**
     * Inserts a widget into the tab panel.
     *
     * @param widget      the widget to be inserted
     * @param tabText     the text to be shown on its tab
     * @param asHTML      <code>true</code> to treat the specified text as HTML
     * @param beforeIndex the index before which it will be inserted
     */
    public void insert(Widget widget, String tabText, boolean asHTML,
                       int beforeIndex) {
        children.insert(widget, beforeIndex);
        qTabBar.insertTab(tabText, asHTML, beforeIndex);
        deck.insert(widget, beforeIndex);
    }

    /**
     * Inserts a widget into the tab panel.
     *
     * @param widget      the widget to be inserted
     * @param tabText     the text to be shown on its tab
     * @param beforeIndex the index before which it will be inserted
     */
    public void insert(Widget widget, String tabText, int beforeIndex) {
        insert(widget, tabText, false, beforeIndex);
    }

    public Iterator iterator() {
        return children.iterator();
    }

    public boolean onBeforeTabSelected(SourcesTabEvents sender, int tabIndex) {
        if(tabListeners != null) {
            return tabListeners.fireBeforeTabSelected(this, tabIndex);
        }
        return true;
    }

    public void onTabSelected(SourcesTabEvents sender, int tabIndex) {
        deck.showWidget(tabIndex);
        if(tabListeners != null) {
            tabListeners.fireTabSelected(this, tabIndex);
        }
    }

    public boolean remove(int index) {
        return remove(getWidget(index));
    }

    /**
     * Removes the given widget, and its associated tab.
     *
     * @param widget the widget to be removed
     */
    public boolean remove(Widget widget) {
        int index = getWidgetIndex(widget);
        if(index == -1) {
            return false;
        }

        children.remove(widget);
        qTabBar.removeTab(index);
        deck.remove(widget);
        return true;
    }

    public void removeTabListener(TabListener listener) {
        if(tabListeners != null) {
            tabListeners.remove(listener);
        }
    }

    /**
     * Programmatically selects the specified tab.
     *
     * @param index the index of the tab to be selected
     */
    public void selectTab(int index) {
        qTabBar.selectTab(index);
    }
}

