/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */
package com.queplix.core.client.common.ui.resizable;

import com.google.gwt.user.client.ui.DeckPanel;
import com.google.gwt.user.client.ui.Widget;
import com.queplix.core.client.common.StringUtil;

/**
 * Resizable deck panel.
 *
 * @author Sultan Tezadov
 * @since 31 Dec 2006
 */
public class ResizableDeckPanel extends DeckPanel implements Resizable {
    private boolean allowNonResizable;

    public ResizableDeckPanel() {
    }

    public void showWidget(int index) {
        super.showWidget(index);
        doSetChildOffsetHeight(index);
        doSetChildOffsetWidth(index);
    }

    protected void add(Widget w, boolean allowNonResizable) {
        this.allowNonResizable = allowNonResizable;
        super.add(w);
    }

    public void insert(Widget w, int beforeIndex) {
        if(!allowNonResizable) { // if it was not set with add() call
            checkResizable(w);
        } else {
            allowNonResizable = false; // reset
        }
        super.insert(w, beforeIndex);
    }

    private void checkResizable(Widget w) {
        if(!(w instanceof Resizable)) {
            throw new IllegalArgumentException(
                    "Illegal argument: w. It must implement Resizable.");
        }
    }

    // ======================== Resizable implementation =====================
    private int offsetHeight;
    private int offsetWidth;

    public void setOffsetHeight(int height) {
        if((height < 0) || (offsetHeight == height)) {
            return; // invalid or unchanged value
        }
        offsetHeight = height;
        if(isAttached()) {
            doSetOffsetHeight();
        }
    }

    public void setOffsetWidth(int width) {
        if((width < 0) || (offsetWidth == width)) {
            return; // invalid or unchanged value
        }
        offsetWidth = width;
        if(isAttached()) {
            doSetOffsetWidth();
        }
    }

    private void doSetOffsetHeight() {
        int index = getVisibleWidget();
        doSetChildOffsetHeight(index);
        super.setHeight(StringUtil.pixelToSize(offsetHeight));
    }

    private void doSetChildOffsetHeight(int index) {
        Resizable widget = (Resizable) getWidget(index);
        if(widget != null) {
            widget.setOffsetHeight(offsetHeight);
        }
    }

    private void doSetOffsetWidth() {
        int index = getVisibleWidget();
        doSetChildOffsetWidth(index);
        super.setWidth(StringUtil.pixelToSize(offsetWidth));
    }

    private void doSetChildOffsetWidth(int index) {
        Resizable widget = (Resizable) getWidget(index);
        if(widget != null) {
            widget.setOffsetWidth(offsetWidth);
        }
    }

    protected void onAttach() {
        super.onAttach();
        if(offsetHeight > 0) { // offsetHeight was set
            doSetOffsetHeight();
        }
        if(offsetWidth > 0) { // offsetWidth was set
            doSetOffsetWidth();
        }
    }

    public void setHeight(String height) {
        setOffsetHeight(StringUtil.sizeToPixel(height));
    }

    public void setWidth(String width) {
        setOffsetWidth(StringUtil.sizeToPixel(width));
    }
    // ===================== End of Resizable implementation ==================

}
