/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */
package com.queplix.core.client.common.ui.resizable;

import com.google.gwt.user.client.ui.StackPanel;
import com.google.gwt.user.client.ui.Widget;
import com.queplix.core.client.common.StringUtil;

/**
 * Resizable stack panel.
 *
 * @author Sultan Tezadov
 * @since 8 Nov 2006
 */
public class ResizableStackPanel extends StackPanel implements Resizable {

    public ResizableStackPanel() {
        super();
    }

    public void add(Widget w, String stackText) {
        this.add(w, stackText, false);
    }

    public void add(Widget w, String stackText, boolean asHTML) {
        super.add(w, stackText, asHTML);
        if(!(w instanceof Resizable)) {
            throw new IllegalArgumentException(
                    "Illegal argument: w. It must implement Resizable.");
        }
    }

    public void showStack(int index) {
        super.showStack(index);
        doSetOffsetHeight();
        doSetOffsetWidth();
    }

    // ======================== Resizable implementation =====================
    private int offsetHeight;
    private int offsetWidth;

    public void setOffsetHeight(int height) {
        if((height < 0) || (offsetHeight == height)) {
            return; // invalid or unchanged value
        }
        offsetHeight = height;
        if(isAttached()) {
            doSetOffsetHeight();
        }
    }

    public void setOffsetWidth(int width) {
        if((width < 0) || (offsetWidth == width)) {
            return; // invalid or unchanged value
        }
        offsetWidth = width;
        if(isAttached()) {
            doSetOffsetWidth();
        }
    }

    private void doSetOffsetHeight() {
        Resizable innerWidget = (Resizable) getWidget(getSelectedIndex());
        innerWidget.setOffsetHeight(offsetHeight);

        super.setHeight(StringUtil.pixelToSize(offsetHeight));
        int actualHeight = getOffsetHeight();
        if(actualHeight != offsetHeight) {
            // correct the height
            int delta = actualHeight - offsetHeight;
            int correctedHeight = offsetHeight - delta;

            innerWidget.setOffsetHeight(correctedHeight);

            if(correctedHeight > 0) {
                super.setHeight(StringUtil.pixelToSize(correctedHeight));
            }
        }
    }

    private void doSetOffsetWidth() {
        Resizable innerWidget = (Resizable) getWidget(getSelectedIndex());
        innerWidget.setOffsetWidth(offsetWidth);

        super.setWidth(StringUtil.pixelToSize(offsetWidth));
        int actualWidth = getOffsetWidth();
        if(actualWidth != offsetWidth) {
            // correct the width
            int delta = actualWidth - offsetWidth;
            int correctedWidth = offsetWidth - delta;

            innerWidget.setOffsetWidth(correctedWidth);

            if(correctedWidth > 0) {
                super.setWidth(StringUtil.pixelToSize(correctedWidth));
            }
        }
    }

    protected void onAttach() {
        super.onAttach();
        if(offsetHeight > 0) { // offsetHeight was set
            doSetOffsetHeight();
        }
        if(offsetWidth > 0) { // offsetWidth was set
            doSetOffsetWidth();
        }
    }

    public void setHeight(String height) {
        setOffsetHeight(StringUtil.sizeToPixel(height));
    }

    public void setWidth(String width) {
        setOffsetWidth(StringUtil.sizeToPixel(width));
    }
    // ===================== End of Resizable implementation ==================

}
