/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */
package com.queplix.core.client.controls.history;

import com.google.gwt.user.client.ui.ClickListener;
import com.google.gwt.user.client.ui.HTML;
import com.google.gwt.user.client.ui.Widget;
import com.queplix.core.client.app.vo.uisettings.DialogUISettings;
import com.queplix.core.client.common.StringUtil;
import com.queplix.core.client.common.ui.ButtonData;
import com.queplix.core.client.common.ui.IconButton;
import com.queplix.core.client.common.ui.OkayCancelPopup;
import com.queplix.core.client.common.ui.resizable.ResizableScrollPanel;
import com.queplix.core.client.controls.memo.QMemo;

import java.util.ArrayList;

/**
 * @author: Melnik Alexander
 * @since: 11.01.2007
 */
class QHistoryViewImpl extends QHistoryView
        implements QHistoryListener, ClickListener {
    private final static int DEFAULT_WIDTH = 400;
    private final static int DEFAULT_HEIGHT = 600;

    private QHistoryModelImpl model;
    private ArrayList eventHandlers;

    private IconButton showbutton;
    private OkayCancelPopup dialog;
    private HTML html;

    private static final ButtonData EMPTY_STATE = new ButtonData(null, "Empty",
            QMemo.BUTTON_EMPTY_IMAGE_PATH);
    private static final ButtonData FILLED_STATE = new ButtonData(null,
            "Filled", QMemo.BUTTON_FULL_IMAGE_PATH);

    public QHistoryViewImpl(QHistoryModelImpl model, int layout) {
        super(model, layout);
        this.model = model;
        eventHandlers = new ArrayList();
        initGUIVisible();
        subscribeEvents();
    }

    public void onModelMetaChanged() {
        setupIndicatorState();
        dataChanged();
        setupDialogTitles();
    }

    public void onModelDataChanged() {
        super.onModelDataChanged();
        dataChanged();
    }

    private void initGUIVisible() {
        // Actual visible part
        showbutton = new IconButton(
                model.getData().isHasHistory() ? FILLED_STATE:EMPTY_STATE);
        addToLabelPanel(showbutton);
        addToLabelPanel(emptyButton);

        initGUIPopup();

        initWidget(panel);
    }

    private void updateButtonState() {
        showbutton.setButtonState(
                model.getData().isHasHistory() ? FILLED_STATE:EMPTY_STATE);
    }

    public void setupDialogTitles() {
        dialog.setCaption(model.getMeta().getPrefixCaption() + " : "
                + model.getData().getRecordId() + " : " + model.getMeta()
                .getCaption());
    }

    public void onDemandModelChanged() {
        model.getData().setText(model.getOnDemandData().getHtml());
        showPopup();
    }

    private void initGUIPopup() {
        if(dialog == null) {
            dialog = new OkayCancelPopup("");
            dialog.addStyleName("memo");

            html = new HTML("");

            ResizableScrollPanel rsp = new ResizableScrollPanel();
            rsp.add(html);
            dialog.setWidget(rsp);
            dialog.setUISettings(model.getMeta().getUISettings());
            rsp.setOffsetHeight(DEFAULT_HEIGHT);
            rsp.setOffsetWidth(DEFAULT_WIDTH);
        }

        html.setHTML(model.getData().getText());

        setupDialogTitles();
    }

    public void showPopup() {
        if(isGoodToShow()) {
            initGUIPopup();
            dialog.show(showbutton);
        }
    }

    private boolean isGoodToShow() {
        return (isDisabled() || isEditMode()) && model.getData().isHasHistory();
    }

    public void addEventHandler(HistoryViewListener handler) {
        eventHandlers.add(handler);
    }

    public void removeEventHandler(HistoryViewListener handler) {
        eventHandlers.remove(handler);
    }

    private void subscribeEvents() {
        showbutton.addClickListener(this);
    }

    public void onClick(Widget sender) {
        if(sender == showbutton) {
            if(isGoodToShow()) {
                for(int i = 0; i < eventHandlers.size(); i++) {
                    HistoryViewListener listener
                            = (HistoryViewListener) eventHandlers.get(i);
                    listener.showDialog();
                }
            }
        }
    }

    protected void onModeChanged(int newMode) {
        showbutton.setEnabled(newMode != MODE_REPORT_DESIGN);
    }

    protected void setEnabled(boolean isEnabled) {
    }

    public void dataChanged() {
        if(!getCaption().equalsIgnoreCase(model.getMeta().getCaption())) {
            setCaption(model.getMeta().getCaption());
        }
        updateButtonState();
    }

    public void setText(String newText) {
        html.setText(StringUtil.nullToEmpty(newText));
    }

    public void setWidth(String width) {
        panel.setWidth(width);
    }

    protected int getButtonWidth() {
        return showbutton.getOffsetWidth() + emptyButton.getOffsetWidth() - 1;
    }

    protected DialogUISettings getUISettings() {
        return (dialog != null) ? dialog.getUISettings():null;
    }
}
