/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */

package com.queplix.core.client.controls.tree;

import java.util.Vector;

/**
 * The class representing a tree node in the tree data concept
 *
 * @author: Vasily Mikhailitchenko
 * @since: 21.09.2006
 */
public class QTreeNode {
    private String string;
    private String icon;
    private QTreeNode parentNode;
    private Vector children;
    private Object id;

    public QTreeNode(String string, QTreeNode parentNode, Object id,
                     String icon) {
        this.string = string;
        this.parentNode = parentNode;
        this.id = id;
        this.setIcon(icon);
        children = new Vector();
    }

    public QTreeNode(String string, Object id, String icon) {
        this(string, null, id, icon);
    }

    public QTreeNode(String string, Object id) {
        this(string, id, null);
    }

    /**
     * Get the text string of a node
     *
     * @return text
     */
    public String getText() {
        return string;
    }

    /**
     * Set node text
     *
     * @param string of text
     */
    public void setText(String string) {
        this.string = string;
    }

    public String getIcon() {
        return this.icon;
    }

    public void setIcon(String icon) {
        this.icon = icon;
    }

    /**
     * Get the parent node
     *
     * @return parent node
     */
    public QTreeNode getParentNode() {
        return parentNode;
    }

    /**
     * Set the parent node
     *
     * @param parent node
     */
    public void setParentNode(QTreeNode parent) {
        this.parentNode = parent;
    }

    /**
     * Add a child to a node
     *
     * @param child node to add
     */
    public void addChild(QTreeNode child) {
        child.setParentNode(this);
        children.add(child);
    }

    /**
     * Remove a child from a node
     *
     * @param child node to remove
     */
    public void removeChild(QTreeNode child) {
        children.remove(child);
    }

    /**
     * Get the node's directs children
     *
     * @return Vector of children
     */
    public Vector getChildren() {
        return children;
    }

    /**
     * Get a child at a particular index
     *
     * @param index - child's index
     * @return the child
     */
    public QTreeNode getChild(int index) {
        try {
            return (QTreeNode) children.get(index);
        }
        catch (Exception e) {
            return null;
        }
    }

    /**
     * Get the index for a particular child
     *
     * @param child - child node
     * @return integer value for index
     */
    public int getIndexOfChild(QTreeNode child) {
        return children.indexOf(child);
    }

    /**
     * Get the number of direct children the node has
     *
     * @return children quantity
     */
    public int getChildCount() {
        return children.size();
    }

    /**
     * Determine if the node is a leaf (has no children)
     * or not
     *
     * @return boolean value
     */
    public boolean isLeaf() {
        if(children.size() == 0) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Get the Id of the node.
     *
     * @return node Id
     */
    public Object getId() {
        return id;
    }
}
