/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */

package com.queplix.core.client.frames.mainframe;

import com.queplix.core.client.app.vo.FamgMeta;
import com.queplix.core.client.app.vo.FieldDataRequest;
import com.queplix.core.client.app.vo.GridSearchProperties;

import java.util.Collection;

/**
 * The base idea of using this interface is to create default implementaion
 * and reuse all code that is possible, but each class, that implements
 * this interface should perform actions based on only on
 * {@link com.queplix.core.client.frames.mainframe.OperationContext} interface
 * and not casting it to wanted class/interface.
 * <p/>
 * This methods interface to be extended with the methods, that performs
 * mainframe, which is necessary to customize in deriving applications, but
 * be carefully in extending, avoid cyclic dependency of this package from
 * concrete application.
 *
 * @author Sergey Kozmin
 * @todo add base description for all functionality
 * @since 10.04.2007
 */
public interface BusinessOperationStrategy {
    /**
     * This method will be called before any other methods will be called.
     *
     * @param operationContext operation context
     * @param formId           current form id and index
     * @param index            current form index
     */
    public void init(OperationContext operationContext, String formId,
                     FamgMeta.Index index);

    /**
     * Create record prototype from the given preferred values.
     *
     * @param data collection of preferred data
     *             Collection<{@link com.queplix.core.client.app.vo.FieldData}>
     */
    public void createRecordProtorype(Collection data);

    /**
     * Lock and return locked version of the record.
     *
     * @param rowID record id
     */
    public void lockAndEditRecord(Long rowID);

    /**
     * Update record with the given data
     *
     * @param rowID row id to be updated
     * @param data  data to be updated.
     *              Collection<{@link com.queplix.core.client.app.vo.FieldData}>
     */
    public void updateRecord(Long rowID, Collection data);

    /**
     * Insert record with the given data
     *
     * @param rowID row id to be inserted
     * @param data  data to be inserted.
     *              Collection<{@link com.queplix.core.client.app.vo.FieldData}>
     */
    public void insertRecord(Long rowID, Collection data);

    /**
     * Delete records with the given ids.
     *
     * @param rowIds row ids to be deleted
     */
    public void deleteRecord(Collection rowIds);

    /**
     * Search records with the given filters and properties.
     *
     * @param entitiesFilters search filters Collection<EntityData>
     * @param props           grid search properties
     * @param isLocalSearch   is local search
     */
    public void searchRecords(Collection entitiesFilters,
                              GridSearchProperties props,
                              boolean isLocalSearch);

    /**
     * This method called when a control needs additional filters
     * @see #handleControlDataRequest(FieldDataRequest)
     * @param request "on demand" request object
     */
    public void addControlCustomFilters(FieldDataRequest request);

    /**
     * This method called when a control needs additional data
     *
     * @param request "on demand" request object
     */
    public void handleControlDataRequest(FieldDataRequest request);

    /**
     * This method handles custom menu events
     *
     * @param eventId menu event id
     */
    public void handleCustomMenuEvent(String eventId);

    /**
     * This method handles customn button event.
     *
     * @param buttonId button id
     */
    public void handleCustomButtonEvent(String buttonId);

    /**
     * Serf from src form to destignation form
     *
     * @param fromLinkFieldId the field id, that can be used to get record id
     */
    public void handleLinkEvent(String fromLinkFieldId);

    /**
     * This method will be called when control event is happend and appropriate
     * even type was set in {@link com.queplix.core.client.frames.mainframe.ViewPropertiesProvider}
     * for the control.
     * @param elementId element id (control id)
     * @param type event type
     */
    public void handleControlEvent(String elementId, FormControlEventTypes type);
}
