/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */

package com.queplix.core.integrator;

import com.queplix.core.client.app.vo.EntityData;
import com.queplix.core.client.app.vo.FieldData;
import com.queplix.core.integrator.entity.EntityFacade;
import com.queplix.core.integrator.entity.RequestProperties;
import com.queplix.core.integrator.security.LogonSession;
import com.queplix.core.modules.eql.error.EQLException;
import com.queplix.core.modules.eqlext.jxb.gr.Reqs;

import java.util.Collection;
import java.util.List;

/**
 * Helps to instantiate appropriate request builder
 *
 * @author Sergey Kozmin
 * @since 05.04.2007
 */
class RequestBuilderFactory {
    public static RequestBuilder createRequestBuilder(
            String requestingEntityName, String filteringEntityName,
            Collection<FieldData> entityFilters, RequestProperties props,
            LogonSession ls, ActionContext ctx) {
        return new FieldsRequestBuilder(
                requestingEntityName, filteringEntityName, entityFilters, props,
                ls, ctx);
    }

    public static RequestBuilder createRequestBuilder(String entityName,
                                                      List<EntityData> entityFilters,
                                                      RequestProperties props,
                                                      LogonSession ls,
                                                      ActionContext ctx) {
        return new EntitiesRequestBuilder(entityName, entityFilters, props, ls,
                ctx);
    }

    public static RequestBuilder createRequestBuilder(
            String requestingEntityName,
            Collection<Long> entityIds,
            RequestProperties props, LogonSession ls, ActionContext ctx) {

        return new IdsRequestBuilder(requestingEntityName, entityIds, props, ls,
                ctx);
    }

    private static class EntitiesRequestBuilder extends BaseRequestBuilder {
        private List<EntityData> entityFilters;
        private String eqlFilter;

        public EntitiesRequestBuilder(
                String entityName, List<EntityData> entityFilters,
                String eqlFilter,
                RequestProperties props, LogonSession ls, ActionContext ctx) {
            super(ls, props, ctx, entityName);
            this.eqlFilter = eqlFilter;
            this.entityFilters = entityFilters;
        }

        public EntitiesRequestBuilder(
                String entityName, List<EntityData> entityFilters,
                RequestProperties props, LogonSession ls, ActionContext ctx) {
            this(entityName, entityFilters, null, props, ls, ctx);
        }

        public Reqs buildRequest(int page) throws EQLException {
            props.setPage(page);
            return EntityFacade.buildReqsFromEntities(entityFilters,
                    eqlFilter,
                    requestingEntityName, props, ls, ctx);
        }
    }

    private static class FieldsRequestBuilder extends BaseRequestBuilder {
        private String filteringEntityName;
        private Collection<FieldData> entityFilters;

        public FieldsRequestBuilder(
                String requestingEntityName,
                String filteringEntityName,
                Collection<FieldData> entityFilters,
                RequestProperties props, LogonSession ls, ActionContext ctx) {
            super(ls, props, ctx, requestingEntityName);
            this.filteringEntityName = filteringEntityName;
            this.entityFilters = entityFilters;
        }

        public Reqs buildRequest(int page) throws EQLException {
            props.setPage(page);
            return EntityFacade.buildReqsFromFields(entityFilters,
                    requestingEntityName, filteringEntityName,
                    null, props, ls, ctx);
        }
    }

    private static class IdsRequestBuilder extends BaseRequestBuilder {
        private Collection<Long> entityIds;

        public IdsRequestBuilder(
                String requestingEntityName,
                Collection<Long> entityIds,
                RequestProperties props, LogonSession ls, ActionContext ctx) {
            super(ls, props, ctx, requestingEntityName);
            this.entityIds = entityIds;
        }

        public Reqs buildRequest(int page) throws EQLException {
            props.setPage(page);
            return EntityFacade.buildRequestByIDs(requestingEntityName,
                    entityIds, props, ls, ctx);
        }
    }
}
