/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */

package com.queplix.qwoss.notification;

import com.queplix.core.integrator.security.LogonSession;
import com.queplix.core.integrator.security.SecurityHelper;
import com.queplix.core.integrator.security.User;
import com.queplix.core.integrator.security.WorkGroup;
import com.queplix.core.jxb.entity.Entity;
import com.queplix.core.modules.config.ejb.EntityViewConfigManagerLocal;
import com.queplix.core.modules.config.ejb.EntityViewConfigManagerLocalHome;
import com.queplix.core.modules.eql.CompoundKey;
import com.queplix.core.modules.eql.error.EQLException;
import com.queplix.core.modules.jeo.ejb.JEOManagerLocal;
import com.queplix.core.modules.jeo.ejb.JEOManagerLocalHome;
import com.queplix.core.utils.JNDINames;
import com.queplix.core.utils.ejb.AbstractSessionEJB;
import com.queplix.qwoss.utils.CustomJNDINames;

import javax.ejb.CreateException;

/**
 * Base class for Notification Manager EJBs.
 *
 * @author [ALB] Baranov Andrey
 * @version $Revision: 1.4.2.1 $ $Date: 2005/12/20 16:43:05 $
 */
public abstract class AbstractEntityNotificationManagerEJB extends AbstractSessionEJB {

    /**
     * User logon session.
     */
    protected LogonSession ls;

    /**
     * Base entity.
     */
    protected Entity entity;

    /**
     * Primary key value object.
     */
    protected CompoundKey key;

    /**
     * JEO Manager EJB local interface reference.
     */
    protected JEOManagerLocal jeoManager;

    /**
     * Initializes EJB.
     *
     * @param ls         LogonSession object
     * @param entityName entity name
     * @param pkey       entity primary key
     */
    public void init(LogonSession ls, String entityName, Long pkey,
                     Integer type) {

        // Check arguments.
        if(ls == null) {
            throw new IllegalStateException("Login session is NULL.");
        }
        if(pkey == null) {
            throw new IllegalStateException("Primary key is NULL.");
        }
        this.ls = ls;
        // Get entity.
        EntityViewConfigManagerLocal local = (EntityViewConfigManagerLocal) 
            getLocalObject( JNDINames.EntityViewConfigManager, EntityViewConfigManagerLocalHome.class);
        entity = local.getEntityViewConfig(entityName);
        // Make a primary key VO.
        key = new CompoundKey();
        key.addKey(String.valueOf(pkey));
        if(type != null) {
            key.addKey(String.valueOf(type));
        }
        // Get JEO Manager EJB reference.
        jeoManager = (JEOManagerLocal) getLocalObject(JNDINames.JEOManager, JEOManagerLocalHome.class);
    }

    public abstract void doNotification();

    /**
     * Loads the user by ID.
     *
     * @param  user ID
     * @return user value object
     */
    protected User loadNotifiedUser(Long userID) {
        try {
            if(userID != null) {
                return SecurityHelper.loadNotifiedUser(ls, userID);
            }
        } catch (EQLException ex) {
            throwException(ex);
        }
        return null;
    }

    /**
     * Loads the group by ID.
     *
     * @param groupID group ID
     * @return group value object
     */
    protected WorkGroup loadNotifiedGroup(Long groupID) {
        try {
            if(groupID != null) {
                return SecurityHelper.loadGroup(ls, groupID);
            }
        } catch (EQLException ex) {
            throwException(ex);
        }
        return null;
    }

    /**
     * Notification Manager EJB local interface getter
     *
     * @return EJB local interface reference
     */
    protected NotificationManagerLocal getNotificationManagerLocal() {
        NotificationManagerLocalHome home = (NotificationManagerLocalHome) 
            getLocalHome( CustomJNDINames.NotificationManager, NotificationManagerLocalHome.class);
        try {
            return home.create(ls, entity, key);
        } catch (CreateException ex) {
            throwException("Can't get NotificationManager local interface", ex);
            return null;
        }
    }

}
