/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */

package com.queplix.qwoss.update;


import com.queplix.core.error.GenericSystemException;
import com.queplix.core.integrator.security.LogonSession;
import com.queplix.core.modules.eql.error.EQLException;
import com.queplix.core.modules.eql.update.EntityUpdate;
import com.queplix.core.modules.jeo.JEObjectHandler;
import com.queplix.core.modules.jeo.ejb.JEOManagerLocal;
import com.queplix.core.modules.jeo.ejb.JEOManagerLocalHome;
import com.queplix.core.utils.JNDINames;
import com.queplix.qwoss.gen.TicketObject;
import com.queplix.qwoss.gen.TicketObjectHandler;
import com.queplix.qwoss.notification.TicketEscalationManagerLocal;
import com.queplix.qwoss.notification.TicketEscalationManagerLocalHome;
import com.queplix.qwoss.notification.TicketNotificationManagerLocal;
import com.queplix.qwoss.notification.TicketNotificationManagerLocalHome;
import com.queplix.qwoss.utils.CustomJNDINames;
import javax.ejb.CreateException;

/**
 * This class tries to perform notification 
 * and generate escalation events for updated ticket.
 *
 * @author rustem.nizamiev
 * @since 18 January 2007, 15:16
 */
public class TicketPostUpdateNotification extends EntityUpdate {

    public void afterUpdate() throws EQLException {

        long time = System.currentTimeMillis();
        INFO("TicketPostUpdateNotification script started ...");
        // Initialization.
        JEOManagerLocal jeoManager = (JEOManagerLocal)
                getCOM().getLocalObject(JNDINames.JEOManager,
                        JEOManagerLocalHome.class);
        LogonSession ls = getEUO().getSession().getLogonSession();
        // Get current ticket record.
        JEObjectHandler hnd = getJEOHandler(jeoManager,
                TicketObjectHandler.class);
        TicketObject ticketObj = (TicketObject) hnd.getJEObject();
        // Send ticket notifications if status or priority changed.
        if(ticketObj.isChangedQw_status() || ticketObj.isChangedQw_priority()) {
            TicketNotificationManagerLocal tnManager
                    = getTicketNotificationManager(ls, ticketObj);
            INFO("Try to do notification for Ticket # "+ticketObj.getQw_ticketid());
            try {
                tnManager.doNotification();
            } finally {
                try {
                    tnManager.remove();
                } catch (Throwable t) {
                }
            }
        }
        // Generate ticket escalation events.
        TicketEscalationManagerLocal teManager = getTicketEscalationManager();
        INFO("Try to generate escalation events for Ticket # "+ticketObj.getQw_ticketid());
        teManager.generateEvents(ls, ticketObj);

        INFO("TicketPostUpdateNotification script completed, time (ms) = " + 
                (System.currentTimeMillis() - time));
    }

    // ----- private methods -----
    
    // Ticket Notification Manager EJB reference getter.
    private TicketNotificationManagerLocal getTicketNotificationManager(
            LogonSession ls, TicketObject ticketObj) {

        TicketNotificationManagerLocalHome home = (TicketNotificationManagerLocalHome) getCOM().
            getLocalHome(CustomJNDINames.TicketNotificationManager ,TicketNotificationManagerLocalHome.class);
        try {
            return home.create(ls, ticketObj);
        } catch (CreateException ex) {
            throw new GenericSystemException( "Can't get TicketNotificationManagerEJB local interface", ex);
        }
    }

    // Ticket (case) Escalation Manager EJB reference getter.
    private TicketEscalationManagerLocal getTicketEscalationManager() {
        INFO("Going To get TicketEscalationManagerEJB local interface");
        return (TicketEscalationManagerLocal) getCOM().
            getLocalObject( CustomJNDINames.TicketEscalationManager, TicketEscalationManagerLocalHome.class);
    }

}
