-- ------------ --
-- Domain model --
-- ------------ --

--
-- SYSTEM tables
--
create table dbo.QX_KEYS (
  TABLENAME        VARCHAR(240)    NOT NULL,
  KEYVALUE        INT        NOT NULL,
constraint PK_QX_KEYS primary key (TABLENAME)
)
go

create table dbo.QX_FIELDINFO (
  TABLE_NAME        VARCHAR(18)    NOT NULL,
  COLUMN_NAME        VARCHAR(18)    NOT NULL,
  COLUMN_TYPE        NUMERIC(2)    NOT NULL,
  ENUM_VALUE        NUMERIC(2),
  ENUM_LABEL        VARCHAR(128)
)
go

create unique index IDX_QX_FIELDINFO on dbo.QX_FIELDINFO (
  TABLE_NAME,
  COLUMN_NAME,
  ENUM_VALUE
)
go

create table dbo.QX_SYS_PROP (
  PKEY            INT    NOT NULL,
  PROP_ID        VARCHAR(255)    NOT NULL,
  PROP_VAL        VARCHAR(255),
  PROP_TEXT        TEXT,
  PROP_DATA        IMAGE,
  NEED_RESTART        NUMERIC                   DEFAULT 0,
  DESCRIPTION        VARCHAR(255),
constraint PK_QX_SYS_PROP primary key (PKEY)
)
go

create table dbo.QX_LANGUAGE (
  PKEY          INT          NOT NULL,
  LANGUAGE_ID   CHAR(2)      NOT NULL,
  NAME          VARCHAR(255) NOT NULL,
constraint PK_QX_LANGUAGE primary key (LANGUAGE_ID)
)
go

create table dbo.QX_TIMEZONE (
  PKEY           INT          NOT NULL,
  TIMEZONE_ID    VARCHAR(255) NOT NULL,
  NAME           VARCHAR(255) NOT NULL,
  OFFSET         INT          NOT NULL,
constraint PK_QX_TIMEZONE primary key (PKEY)
)
go

create table dbo.QX_COUNTRY (
  PKEY                 INT          NOT NULL,
  COUNTRY_ID           VARCHAR(3)   NOT NULL,
  NAME                 VARCHAR(255) NOT NULL,
  DEF_LANG_ID          CHAR(2),
  DEF_TIMEZONE_ID      VARCHAR(255),
constraint PK_QX_COUNTRY primary key (PKEY),
--constraint FK_QX_COUNTRY_QX_LANGUAGE    foreign key (DEF_LANG_ID)    references QX_LANGUAGE (LANGUAGE_ID),
--constraint FK_QX_COUNTTRY_QX_TIMEZONE    foreign key (DEF_TIMEZONE_ID)    references QX_TIMEZONE (TIMEZONE_ID)
)
go

create index IDX_QX_COUNTRY_LANGUAGE on dbo.QX_COUNTRY (DEF_LANG_ID)
create index IDX_QX_COUNTRY_TIMEZONE on dbo.QX_COUNTRY (DEF_TIMEZONE_ID)
go

CREATE TABLE dbo.QX_TIME_PATTERN (
  PKEY             INT          NOT NULL,
  TIME_PATTERN_ID  VARCHAR(255) NOT NULL
constraint PK_QX_TIME_PATTERN primary key (PKEY),
)
GO

CREATE TABLE dbo.QX_DATE_PATTERN (
  PKEY             INT          NOT NULL,
  DATE_PATTERN_ID  VARCHAR(255) NOT NULL
constraint PK_QX_DATE_PATTERN primary key (PKEY),
)
GO

create view dbo.QX_DD_DATEPOS as
    select
        ENUM_VALUE as PKEY,
        ENUM_LABEL as NAME
    from QX_FIELDINFO
    where (TABLE_NAME = 'qx_user_settings') and (COLUMN_NAME = 'date_position')
go

create view dbo.QX_DD_YESNO as 
    select
        ENUM_VALUE as PKEY,
        ENUM_LABEL as NAME
    from QX_FIELDINFO
    where (TABLE_NAME = '_FAKE_YES_NO') and (COLUMN_NAME = 'FAKE_COLUMN')
go

--
-- ENTITY/FOCUS
--

create table dbo.QX_ENTITY (
  ENTITY_ID            VARCHAR(255)    NOT NULL,
  ENTITYDATA           IMAGE,
constraint PK_QX_ENTITY primary key (ENTITY_ID)
)
go

--
-- SECURITY
--

create table dbo.QX_VIEW_OBJECT_TYPES (
    pkey                int             not null,
    type_description    varchar(255)    not null,

    constraint PK_QX_VIEW_OBJECT_TYPES primary key (pkey)
)
go

create table dbo.QX_VIEW_OBJECTS (
    pkey                        int             not null,
    name                        varchar(255)    not null,
    type                        int             not null,
    parent_object               int             null,
    order_in_group              int             not null default 0,
    icon                        varchar(255)    null,
    inframelinks				bit				not null default 0,
    grid				        bit				not null default 1,

    constraint PK_QX_VIEW_OBJECTS primary key (pkey),
    constraint FK_QX_VIEW_OBJECTS_TO_OBJECT_TYPES foreign key (type) references QX_VIEW_OBJECT_TYPES(pkey),
    constraint FK_QX_VIEW_OBJECTS_TO_PARENT_OBJECT foreign key (parent_object) references QX_VIEW_OBJECTS(pkey)
)
go

create table dbo.QX_FORM_SETTINGS (
    view_object_id          int         not null,
    entity_id               varchar(255),
    grid                    int,
    use_default_actions     bit         default 1,
    labels_layout           bit         default 0,
    myqueweb           	    bit         not null     default 1,
    autosearch           	bit         not null     default 0,

--    constraint PK_QX_FORM_SETTINGS primary key (view_object_id),
    constraint FK_QX_FORM_SETTINGS_TO_VIEW_OBJECTS foreign key (view_object_id) references QX_VIEW_OBJECTS(pkey)
)
go

create table dbo.QX_ROLE (
    role_id           int             not null,
    default_focus_id  int             null,
    name              varchar(255)    not null,
    description       varchar(255)    null,

    constraint PK_QX_ROLE primary key (role_id),
    constraint FK_QX_ROLE_TO_FOCUS foreign key (default_focus_id) references QX_VIEW_OBJECTS(pkey)
)
go

create table dbo.QX_ACCESSLEVEL (
    pkey    int             not null,
    name    varchar(255)    not null,

    constraint PK_QX_ACCESSLEVEL primary key (pkey)
)
go

create table dbo.QX_PERMISSION (
    permission_id           int              not null,
    role_id                 int              not null,
    access_level            int              not null,
    permission_object_type  int              not null,
    object_id               int              not null,

    constraint PK_QX_PERMISSION primary key (permission_id),
    constraint FK_QX_PERMISSION_TO_ACCESS_LEVEL foreign key (access_level) references qx_accesslevel(pkey),
    constraint FK_QX_PERMISSION_TO_ROLE foreign key (role_id) references qx_role(role_id),
    constraint FK_QX_PERMISSION_TO_PERMISSION_OBJECT_TYPES foreign key (permission_object_type) references QX_VIEW_OBJECT_TYPES(pkey),
    constraint FK_QX_PERMISSION_TO_QX_VIEW_OBJECTS foreign key (object_id) references QX_VIEW_OBJECTS(pkey)
)
go

create table dbo.QX_USER_TYPE (
    USER_TYPE_ID         INT            NOT NULL,
    NAME                 VARCHAR(255)   NOT NULL,

    constraint PK_QX_USER_TYPE primary key (USER_TYPE_ID)
)
go

create table dbo.QX_USERS (
    PKEY                    INT             NOT NULL,
    LOGINNAME               VARCHAR(255)    NOT NULL,
    FULLNAME                VARCHAR(255)    NULL,
    PASSWORD                VARCHAR(255)    NULL,
    EMAIL                   VARCHAR(255)    NULL,
    USER_TYPE               INT             NOT NULL,

    constraint PK_QX_USERS primary key (PKEY),
    constraint FK_QX_USERS_TO_USERTYPE foreign key (USER_TYPE) references qx_user_type(USER_TYPE_ID)
)
go

create unique index IDX_USERS_LOGINNAME on dbo.QX_USERS (LOGINNAME)
go


create table dbo.QX_WORKGROUP (
    pkey                int             not null,
    name                varchar(40)     not null,
    description         varchar(80),
    notificationaddr    varchar(40),
    notifymethod        numeric(2),

    constraint PK_QX_WORKGROUP primary key (PKEY)
)
go

create table dbo.QX_GROUPMEMBER (
  PKEY              INT     NOT NULL,
  EMPLOYEEGROUP     INT     NOT NULL,
  WORKGROUPGROUP    INT     NOT NULL,
  NOTIFYMETHOD      NUMERIC(2),
  TIER              INT     DEFAULT 0,

constraint PK_QX_GROUPMEMBER primary key (PKEY),
constraint FK_QX_GROUPMEMBER_EMPLOYEE   foreign key (EMPLOYEEGROUP)     references QX_USERS(PKEY),
constraint FK_QX_GROUPMEMBER_WORKGROUP  foreign key (WORKGROUPGROUP)    references QX_WORKGROUP(PKEY)
)
go

create index IDX_QX_GROUPMEMBER_EMPLOYEE  on dbo.QX_GROUPMEMBER (EMPLOYEEGROUP)
create index IDX_QX_GROUPMEMBER_WORKGROUP on dbo.QX_GROUPMEMBER (WORKGROUPGROUP)
go

create table dbo.QX_ROLE_TO_USERS (
    pkey                int             not null,
    user_id             int             not null,
    role_id             int             not null,

    constraint FK_QX_ROLE_TO_USER_TO_FIRST_ROLE_FK foreign key (role_id) references QX_ROLE(role_id),
    constraint FK_QX_ROLE_TO_USER_TO_SECOND_USER_FK foreign key (user_id) references QX_USERS(pkey)
)
go

create table dbo.QX_WORKGROUP_TO_ROLE (
    pkey                int             not null,
    role_id             int             not null,
    workgroup_id        int             not null,

    constraint FK_QX_WORKGROUP_TO_ROLE_TO_FIRST_WORKGROUP_FK foreign key (workgroup_id) references QX_WORKGROUP(pkey),
    constraint FK_QX_WORKGROUP_TO_ROLE_TO_SECOND_ROLE_FK foreign key (role_id) references QX_ROLE(role_id)
)
go

create view dbo.qx_user_permissions as
    select u.pkey as user_id, p.*, vo.name as object_name
    from qx_permission p
		inner join qx_role_to_users r2u on p.role_id = r2u.role_id
		inner join qx_users u on u.pkey = r2u.user_id
		inner join qx_view_objects vo on p.object_id = vo.pkey
    union
    select u.pkey as user_id, p.*, vo.name as object_name
    from qx_permission p
		inner join qx_workgroup_to_role w2r on p.role_id = w2r.role_id
		inner join qx_workgroup w on w.pkey = w2r.workgroup_id
		inner join qx_groupmember gm on gm.workgroupgroup = w.pkey
		inner join qx_users u on u.pkey = gm.employeegroup
		inner join qx_view_objects vo on p.object_id = vo.pkey
go

--
-- USERS, WORKGROUPS, ROLES
--
create view dbo.QX_DD_WORKGROUPNOTIFYMETHOD as
    select
        enum_value as PKEY,
        enum_label as NAME
    from qx_fieldinfo
    where (table_name = 'qx_workgroup') and (column_name = 'notifymethod')
go

create view dbo.QX_DD_GROUPMEMBERNOTIFYMETHOD as
    select
        ENUM_VALUE as PKEY,
        ENUM_LABEL as NAME
    from QX_FIELDINFO
    where (TABLE_NAME = 'qx_groupmember') and (COLUMN_NAME = 'notifymethod')
go

create table dbo.QX_USER_PROP (
  PROP_ID              VARCHAR(512)    NOT NULL,
  USER_ID              INT        NOT NULL,
  USER_TYPE_ID         INT        NOT NULL,
  PROP_VAL             VARCHAR(2048),
constraint PK_QX_USER_PROP primary key (PROP_ID, USER_ID, USER_TYPE_ID),
constraint FK_QX_USER_PROP_QX_USER foreign key (USER_TYPE_ID) references QX_USER_TYPE (USER_TYPE_ID)
)
go

create index IDX_QX_USER_PROP_USER      on dbo.QX_USER_PROP (USER_ID)
create index IDX_QX_USER_PROP_USER_TYPE on dbo.QX_USER_PROP (USER_TYPE_ID)
go

create table dbo.QX_USER_SETTINGS (
  PKEY        INT  NOT NULL,
  USER_ID     INT  NOT NULL,
  LANG        INT  NULL,
  COUNTRY     INT  NULL,
  TIMEZONE    INT  NULL,
  DPATTERN    INT  NULL,
  TPATTERN    INT  NULL,
  DPOS        INT  NULL,
constraint PK_USER_SETTINGS primary key (PKEY)
)
go

create unique index IDX_USER_SETTINGS_USER_ID on dbo.QX_USER_SETTINGS (USER_ID)
go


--
-- View
--
create table dbo.QX_HEADER (
  FORM_ID              VARCHAR(255) NOT NULL,
  HEADER_ORDER         INT NOT NULL,
  CLIENT_WIDTH         INT,
constraint PK_QX_HEADER primary key (FORM_ID, HEADER_ORDER)
)

create table dbo.QX_ROW (
  FORM_ID              VARCHAR(255) NOT NULL,
  ROW_ORDER            INT NOT NULL,
constraint PK_QX_ROW primary key (FORM_ID, ROW_ORDER)
)

create table dbo.QX_COL (
  ROW_ORDER            VARCHAR(255) NOT NULL,
  FORM_ID              VARCHAR(255) NOT NULL,
  COL_ORDER            INT NOT NULL,
  FIELD_ID             VARCHAR(255),
  ROWSPAN              INT,
  COLSPAN              INT,
constraint PK_QX_COL primary key (ROW_ORDER, FORM_ID, COL_ORDER)
)  

create table dbo.QX_BUTTON (
  FORM_ID              VARCHAR(255) NOT NULL,
  BUTTON_ID            VARCHAR(255) NOT NULL,
  BUTTON_ORDER         INT,
  PERMISSION           INT,
constraint PK_QX_BUTTON primary key (FORM_ID, BUTTON_ID)
)

create table dbo.QX_HTML_ELEMENT (
  FORM_ID              VARCHAR(255) NOT NULL,
  HTML_ELEMENT_ID      VARCHAR(255) NOT NULL,
  HTML_ELEMENT_ORDER   INT,
constraint PK_QX_HTML_ELEMENT primary key (FORM_ID, HTML_ELEMENT_ID)
)

create table dbo.QX_HIDDENCONTROL (
  FORM_ID              VARCHAR(255) NOT NULL,
  FIELD_ID            VARCHAR(255) NOT NULL,
constraint PK_QX_HIDDENCONTROL primary key (FORM_ID, FIELD_ID)
)

create table dbo.QX_LINK (
  FORM_ID              VARCHAR(255) NOT NULL,
  FIELD_ID             VARCHAR(255) NOT NULL,
  LINKED_FORM_ID       VARCHAR(255) NOT NULL,
constraint PK_QX_LINK primary key (FORM_ID, FIELD_ID)
)

create table dbo.QX_EXTERNALFIELD (
  EXTERNALFORM_ID      VARCHAR(255)    NOT NULL,
  FORM_ID              VARCHAR(255)    NOT NULL,
  EXTERNALFIELD        VARCHAR(255),
  SOURCEFIELD          VARCHAR(255),
constraint PK_QX_EXTERNALFIELD primary key (EXTERNALFORM_ID, FORM_ID)
)
go

create index IDX_QX_EXTERNALFIELD_FORM  on dbo.QX_EXTERNALFIELD (FORM_ID)
create index IDX_QX_EXTERNALFIELD_FIELD on dbo.QX_EXTERNALFIELD (EXTERNALFIELD)
go

create table dbo.QX_EXTERNALFORM (
  EXTERNALFORM_ID      VARCHAR(255)    NOT NULL,
  FORM_ID              VARCHAR(255)    NOT NULL,
constraint PK_QX_EXTERNALFORM primary key (EXTERNALFORM_ID, FORM_ID)
)
go

create index IDX_QX_EXTERNALFORM_FORM on dbo.QX_EXTERNALFORM (FORM_ID)
go



create table dbo.QX_EXTERNALSET (
  EXTERNALFORM_ID      VARCHAR(255)    NOT NULL,
  FORM_ID              VARCHAR(255)    NOT NULL,
constraint PK_QX_EXTERNALSET primary key (EXTERNALFORM_ID, FORM_ID)
)
go

create index IDX_QX_EXTERNALSET_FORM on dbo.QX_EXTERNALSET (FORM_ID)
go



create table dbo.QX_LINKEDDATASET (
  ENTITY_ID            VARCHAR(255)    NOT NULL,
  DATASET_ID           VARCHAR(255)    NOT NULL,
  FORM_ID              VARCHAR(255)    NOT NULL,
constraint PK_QX_LINKEDDATASET primary key (ENTITY_ID, DATASET_ID)
)
go

create index IDX_QX_LINKEDDATASET_FORM on dbo.QX_LINKEDDATASET (FORM_ID)
go


--
-- Localization
--

create table dbo.QX_LOCALIZATION_TYPE (
  LOCALIZATION_TYPE_ID   INT             NOT NULL,
  NAME                   VARCHAR(255)    NOT NULL,
constraint PK_QX_LOCALIZATION_TYPE primary key (LOCALIZATION_TYPE_ID)
)
go

create table dbo.QX_LOCALIZED_OBJECT_TYPE (
  OBJECT_TYPE_ID  INT               NOT NULL,
  NAME            VARCHAR(255)      NOT NULL,
constraint PK_QX_LOCALIZED_OBJECT_TYPE primary key (OBJECT_TYPE_ID)
)
go

create table dbo.QX_LOCALIZATION (
  LOCALIZATION_ID	    VARCHAR(255)      NOT NULL,
  LOCALIZATION_TYPE_ID  INT               NOT NULL,
  OBJECT_TYPE_ID	    INT               NOT NULL,
  LANGUAGE_ID		    CHAR(2)           NOT NULL,
  CONTENT               NVARCHAR(4000)    NOT NULL,
constraint PK_QX_LOCALIZATION primary key (LOCALIZATION_ID, LOCALIZATION_TYPE_ID, LANGUAGE_ID, OBJECT_TYPE_ID),
constraint FK_QX_LOCALIZATION_QX_LANGUAGE foreign key (LANGUAGE_ID) references QX_LANGUAGE (LANGUAGE_ID),
constraint FK_QX_LOCALIZATION_QX_LOCALIZATION_TYPE foreign key (LOCALIZATION_TYPE_ID) references QX_LOCALIZATION_TYPE (LOCALIZATION_TYPE_ID),
constraint FK_QX_LOCALIZATION_QX_LOCALIZED_OBJECT_TYPE foreign key (OBJECT_TYPE_ID) references QX_LOCALIZED_OBJECT_TYPE (OBJECT_TYPE_ID)
)
go

create index IDX_QX_LOCALIZATION_LANGUAGE on dbo.QX_LOCALIZATION (LANGUAGE_ID)
-- [MVT] WARNING! Don't try to create index on the
--                LOCALIZATION_TYPE_ID column - appication hangs
--create index IDX_QX_LOCALIZATION_TYPE on dbo.QX_LOCALIZATION (LOCALIZATION_TYPE_ID)
create index IDX_QX_LOCALIZATION_OBJECT on dbo.QX_LOCALIZATION (OBJECT_TYPE_ID)
go


--
-- Context menu
--

create table dbo.QX_CONTEXTMENU (
  CONTEXTMENU_ID    VARCHAR(255)    NOT NULL,
constraint PK_QX_CONTETXMENU primary key (CONTEXTMENU_ID)
)
go

create table dbo.QX_MENUITEM (
  MENUITEM_ID        VARCHAR(255)    NOT NULL,
  CONTEXTMENU_ID    VARCHAR(255)    NOT NULL,
  MENUITEM_ORDER    INT,
constraint PK_QX_MENUITEM primary key (MENUITEM_ID)
)
go

create index IDX_QX_MENUITEM_CONTEXTMENU on dbo.QX_MENUITEM (CONTEXTMENU_ID)
create index IDX_QX_MENUITEM_ORDER on dbo.QX_MENUITEM (MENUITEM_ORDER)
go

create table dbo.QX_CONTEXTMENU_FORM (
  FORM_ID            VARCHAR(255)    NOT NULL,
  CONTEXTMENU_ID    VARCHAR(255)    NOT NULL,
constraint PK_QX_CONTETXMENU_FORM primary key (FORM_ID, CONTEXTMENU_ID)
)
go

--
-- ALERTS
--

create table dbo.QX_ALERT (
  ALERT_ID             INT        NOT NULL,
  SENDER_ID            INT        NOT NULL,
  SENDER_TYPE          INT        NOT NULL,
  RECIPIENT_ID         INT,
  RECIPIENT_TYPE       INT,
  WORKGROUP_ID         INT,
  TIER                 NUMERIC(2),
  TO_ALL               NUMERIC(2)               DEFAULT 0,
  FOCUS_ID             VARCHAR(255),
  TAB_ID               VARCHAR(255),
  FORM_ID              VARCHAR(255),
  RECORD_ID            VARCHAR(100),
  RECORD_ID2           VARCHAR(100),
  RECORD_ID3           VARCHAR(100),
  RECORD_ID4           VARCHAR(100),
  MESSAGE              VARCHAR(1022)    NOT NULL,
  SEVERITY             NUMERIC(2)    NOT NULL,
  DATEPOSTED           DATETIME        NOT NULL,
  CREATOR_ID           INT        NOT NULL,
  CREATOR_TYPE         INT        NOT NULL,
constraint PK_QX_ALERT primary key (ALERT_ID),
constraint FK_QX_ALERT_QX_USER_TYPE1    foreign key (SENDER_TYPE)    references QX_USER_TYPE (USER_TYPE_ID),
constraint FK_QX_ALERT_QX_USER_TYPE2    foreign key (RECIPIENT_TYPE)    references QX_USER_TYPE (USER_TYPE_ID),
constraint FK_QX_ALERT_QX_USER_TYPE3    foreign key (CREATOR_TYPE)    references QX_USER_TYPE (USER_TYPE_ID)
)
go

create index IDX_QX_ALERT_SENDER         on dbo.QX_ALERT (SENDER_ID)
create index IDX_QX_ALERT_SENDER_TYPE    on dbo.QX_ALERT (SENDER_TYPE)
create index IDX_QX_ALERT_RECIPIENT      on dbo.QX_ALERT (RECIPIENT_ID)
create index IDX_QX_ALERT_RECIPIENT_TYPE on dbo.QX_ALERT (RECIPIENT_TYPE)
create index IDX_QX_ALERT_CREATOR        on dbo.QX_ALERT (CREATOR_ID)
create index IDX_QX_ALERT_CREATOR_TYPE   on dbo.QX_ALERT (CREATOR_TYPE)
create index IDX_QX_ALERT_WORKGROUP      on dbo.QX_ALERT (WORKGROUP_ID)
create index IDX_QX_ALERT_FOCUS          on dbo.QX_ALERT (FOCUS_ID)
create index IDX_QX_ALERT_TAB            on dbo.QX_ALERT (TAB_ID)
create index IDX_QX_ALERT_FORM           on dbo.QX_ALERT (FORM_ID)
go



create table dbo.QX_ALERT_BLOCK (
  ALERT_BLOCK_ID       INT        NOT NULL,
  ALERT_ID             INT        NOT NULL,
  RECIPIENT_ID         INT        NOT NULL,
  RECIPIENT_TYPE       INT        NOT NULL,
constraint PK_QX_ALERT_BLOCK primary key (ALERT_BLOCK_ID),
constraint FK_QX_ALERT_BLOCK_QX_ALERT        foreign key (ALERT_ID)        references QX_ALERT (ALERT_ID),
constraint FK_QX_ALERT_BLOCK_QX_USER_TYPE    foreign key (RECIPIENT_TYPE)    references QX_USER_TYPE (USER_TYPE_ID)
)
go

create unique index IDX_ALERT_BLOCK on dbo.QX_ALERT_BLOCK (
  ALERT_ID,
  RECIPIENT_ID,
  RECIPIENT_TYPE
)
go

create index IDX_QX_ALERT_BLOCK_RECIPIENT      on dbo.QX_ALERT_BLOCK (RECIPIENT_ID)
create index IDX_QX_ALERT_BLOCK_RECIPIENT_TYPE on dbo.QX_ALERT_BLOCK (RECIPIENT_TYPE)
go



create view dbo.QX_DD_ALERTSEVERITY as 
    select
        ENUM_VALUE as PKEY,
        ENUM_LABEL as NAME
    from QX_FIELDINFO
    where (TABLE_NAME = 'qx_alert') and (COLUMN_NAME = 'severity')
go



--
-- ATTACHMENTS
--

create table dbo.QX_ATTACHMENT (
  ATTACHMENT_ID        INT        NOT NULL,
  PROCESS_ID        BIGINT,
  FILENAME        VARCHAR(255) NOT NULL,
  CREATED        DATETIME    NOT NULL,
  INDEXED        CHAR(1),
  DESCRIPTION        VARCHAR(250),
  FILETYPE        VARCHAR(50)               DEFAULT 'application/octet-stream',
  FILEEXT        VARCHAR(10),
  DATA            IMAGE,
  USER_ID        INT,
constraint PK_QX_ATTACHMENT primary key (ATTACHMENT_ID),
)
go

create index IDX_QX_ATTACHMENT_USER      on dbo.QX_ATTACHMENT (USER_ID)
go



create table dbo.QX_TEMP_ATTACH (
  TEMP_ATTACH_ID    INT        NOT NULL,
  PROCESS_ID        BIGINT,
  FILENAME        VARCHAR(255),
  CREATED        DATETIME    NOT NULL,
  DATA            IMAGE,
  FILETYPE        VARCHAR(50)               DEFAULT 'application/octet-stream',
  FILEEXT        VARCHAR(10),
  USER_TYPE_ID        INT,
  USER_ID        INT,
constraint PK_QX_TEMP_ATTACH primary key (TEMP_ATTACH_ID),
constraint FK_QX_TEMP_ATTACH_QX_USER_TYPE foreign key (USER_TYPE_ID) references QX_USER_TYPE (USER_TYPE_ID)
)
go

create index IDX_QX_TEMP_ATTACH_USER_TYPE on dbo.QX_TEMP_ATTACH (USER_TYPE_ID)
create index IDX_QX_TEMP_ATTACH_USER      on dbo.QX_TEMP_ATTACH (USER_ID)
go

create table dbo.QX_ATTACHMENT_OBJECTS (
  PKEY             INT NOT NULL,
  ATTACHMENT_ID    INT,
  OBJECT_ID        INT,
  OBJECT_TYPE      INT,
constraint PK_QX_ATTACHMENT_OBJECTS primary key (PKEY),
constraint FK_QX_ATTACHMENT_OBJECTS_QX_ATTACHMENT foreign key (ATTACHMENT_ID) references QX_ATTACHMENT (ATTACHMENT_ID)
)
go


--
-- IMAGES
--

create table dbo.QX_IMAGE (
  IMAGE_ID             INT        NOT NULL,
  FILE_SIZE            INT,
  CREATED              DATETIME        NOT NULL,
  MODIFIER_ID          INT        NOT NULL,
  MODIFIED             DATETIME        NOT NULL,
  FILENAME             VARCHAR(255)    NOT NULL,
  FOLDER               VARCHAR(255),
  CREATOR_ID           INT        NOT NULL,
constraint PK_QX_IMAGE primary key (IMAGE_ID),
)
go

create index IDX_QX_IMAGE_MODIFIER      on dbo.QX_IMAGE (MODIFIER_ID)
create index IDX_QX_IMAGE_CREATOR       on dbo.QX_IMAGE (CREATOR_ID)
go



create table dbo.QX_IMAGE_FOLDER (
  FOLDER_ID            INT        NOT NULL,
  FOLDER               VARCHAR(255)    NOT NULL,
constraint PK_QX_IMAGE_FOLDER primary key (FOLDER_ID)
)
go



create table dbo.QX_IMAGE_USAGE (
  IMAGE_USAGE_ID       INT        NOT NULL,
  IMAGE_ID             INT        NOT NULL,
  RECORD_KEY           INT        NOT NULL,
  TABLE_NAME           VARCHAR(50)    NOT NULL,
constraint PK_QX_IMAGE_USAGE primary key (IMAGE_USAGE_ID),
constraint FK_QX_IMAGE_USAGE_QX_IMAGE foreign key (IMAGE_ID) references QX_IMAGE (IMAGE_ID)
)
go

create index IDX_QX_IMAGE_USAGE_IMAGE on dbo.QX_IMAGE_USAGE (IMAGE_ID)
go



--
-- LOGGING
--

create table dbo.QX_LOG_TYPE (
  LOG_TYPE_ID          INT        NOT NULL,
  NAME                 VARCHAR(50)    NOT NULL,
constraint PK_QX_LOG_TYPE primary key (LOG_TYPE_ID)
)
go



create table dbo.QX_LOG_CLASS (
  LOG_CLASS_ID         INT        NOT NULL,
  NAME                 VARCHAR(50)    NOT NULL,
  TABLE_NAME           VARCHAR(50)    NOT NULL,
constraint PK_QX_LOG_CLASS primary key (LOG_CLASS_ID)
)
go



create table dbo.QX_LOG (
  LOG_ID               INT        NOT NULL,
  LOG_PROCESS_ID       BIGINT        NOT NULL,
  LOG_TYPE_ID          INT        NOT NULL,
  LOG_CLASS_ID         INT        NOT NULL,
  USER_TYPE_ID         INT        NOT NULL,
  USER_ID              INT        NOT NULL,
  DESCRIPTION          VARCHAR(1022),
  STAMP                DATETIME        NOT NULL,
constraint PK_QX_LOG primary key (LOG_ID),
constraint FK_QX_LOG_QX_USER_TYPE    foreign key (USER_TYPE_ID)    references QX_USER_TYPE (USER_TYPE_ID),
constraint FK_QX_LOG_QX_LOG_TYPE    foreign key (LOG_TYPE_ID)    references QX_LOG_TYPE (LOG_TYPE_ID),
constraint FK_QX_LOG_QX_LOG_CLASS    foreign key (LOG_CLASS_ID)    references QX_LOG_CLASS (LOG_CLASS_ID)
)
go

create index IDX_QX_LOG_USER_TYPE on dbo.QX_LOG (USER_TYPE_ID)
create index IDX_QX_LOG_USER      on dbo.QX_LOG (USER_ID)
create index IDX_QX_LOG_LOG_TYPE  on dbo.QX_LOG (LOG_TYPE_ID)
create index IDX_QX_LOG_LOG_CLASS on dbo.QX_LOG (LOG_CLASS_ID)
go



create table dbo.QX_ERROR_CFG (
  ERROR_CFG_ID        INT        NOT NULL,
  TARGET_CLASS_MASK    VARCHAR(255)    NOT NULL,
  MESSAGE_HND_CLASS    VARCHAR(255)    NOT NULL,
  EMAILS        VARCHAR(1024),
  EMPLOYEE_ID        INT,
  WORKGROUP_ID        INT,
  DISABLE        NUMERIC(2)               DEFAULT 0,
constraint PK_QX_ERROR_CFG primary key (ERROR_CFG_ID)
)
go



create table dbo.QX_USER_LOG (
  USER_LOG_ID          INT        NOT NULL,
  USER_ID              INT        NOT NULL,
  USER_TYPE_ID         INT        NOT NULL,
  SESSION_ID           VARCHAR(255)    NOT NULL,
  FULLNAME             VARCHAR(255),
  HOST                 VARCHAR(255),
  REMOTE_ADDRESS       VARCHAR(15),
  LOGIN_TIME           DATETIME        NOT NULL,
  LOGOUT_TIME          DATETIME,
constraint PK_QX_USER_LOG primary key (USER_LOG_ID),
constraint FK_QX_USER_LOG_QX_USER_TYPE foreign key (USER_TYPE_ID) references QX_USER_TYPE (USER_TYPE_ID)
)
go

create index IDX_QX_USER_LOG_USER      on dbo.QX_USER_LOG (USER_ID)
create index IDX_QX_USER_LOG_USER_TYPE on dbo.QX_USER_LOG (USER_TYPE_ID)
go

--
-- REPORTS
--

create table dbo.QX_REPORT (
    REPORT_ID        INT        NOT NULL,
    NAME            VARCHAR(255)    NOT NULL,
    OWNER_ID        INT,
    PUBLIC_REPORT        NUMERIC(2)               DEFAULT 0,
    TIMESTAMP        DATETIME    NOT NULL,
    BODY            TEXT        NOT NULL,
    constraint PK_QX_REPORT primary key (REPORT_ID)
)
go

create index IDX_QX_REPORT_OWNER         on dbo.QX_REPORT (OWNER_ID)
create index IDX_QX_REPORT_PUBLIC_REPORT on dbo.QX_REPORT (PUBLIC_REPORT)
go

create table dbo.QX_REPORT_SCHED (
  REPORT_SCHED_ID    INT        NOT NULL,
  REPORT_ID        INT        NOT NULL,
  PERIOD        INT        NOT NULL,
  PERIOD_UNIT        INT        NOT NULL,
  ACTIVE        NUMERIC(2)               DEFAULT 1,
  CREATED_BY        INT,
  CREATOR_TYPE        INT,
  NEXT_DATE        DATETIME,
  TO_ADDR        VARCHAR(255)    NOT NULL,
  CC_ADDR        VARCHAR(255),
  SUBJECT        VARCHAR(255)    NOT NULL,
constraint PK_QX_REPORT_SCHED primary key (REPORT_SCHED_ID),
constraint FK_QX_REPSCHED_QX_USER_TYPE foreign key (CREATOR_TYPE) references QX_USER_TYPE (USER_TYPE_ID)
)
go

create index IDX_REPORT_SCHED on dbo.QX_REPORT_SCHED (
  NEXT_DATE,
  ACTIVE
)
go

create index IDX_QX_REPORT_SCHED_REPORT       on dbo.QX_REPORT_SCHED (REPORT_ID)
create index IDX_QX_REPORT_SCHED_CREATED_BY   on dbo.QX_REPORT_SCHED (CREATED_BY)
create index IDX_QX_REPORT_SCHED_CREATOR_TYPE on dbo.QX_REPORT_SCHED (CREATOR_TYPE)
go



create view dbo.QX_DD_REPORT_SCHED_PERIODUNITS AS 
    select
        ENUM_VALUE as PKEY,
        ENUM_LABEL as NAME
    from QX_FIELDINFO
    where (TABLE_NAME = 'qx_report_sched') and (COLUMN_NAME = 'period_unit')
go


--
-- CHARTS
--

create table dbo.QX_CHART (
    CHART_ID        INT             NOT NULL,
    NAME            VARCHAR(255)    NOT NULL,
    OWNER_ID        INT,
    PUBLIC_CHART    INT             NOT NULL  DEFAULT 0,
    SYSTEM_CHART    INT             NOT NULL  DEFAULT 0,
    CHART_POSITION  INT,
    TIMESTAMP       DATETIME        NOT NULL  DEFAULT getdate(),
    BODY            VARCHAR(2048)   NOT NULL,
    constraint PK_QX_CHART primary key (CHART_ID)
)
go

create index IDX_QX_CHART_OWNER         on dbo.QX_CHART (OWNER_ID)
create index IDX_QX_CHART_PUBLIC_CHART on dbo.QX_CHART (PUBLIC_CHART)
create index IDX_QX_CHART_SYSTEM_CHART on dbo.QX_CHART (SYSTEM_CHART)
go


--
-- HISTORY
--

create table dbo.QX_HIS_TABLE (
  HIS_TABLE_ID         INT        NOT NULL,
  NAME                 VARCHAR(255)    NOT NULL,
constraint PK_QX_HIS_TABLE primary key (HIS_TABLE_ID)
)
go


create table dbo.QX_HIS_FIELD (
  HIS_FIELD_ID         INT        NOT NULL,
  HIS_TABLE_ID         INT        NOT NULL,
  NAME                 VARCHAR(255)    NOT NULL,
constraint PK_QX_HIS_FIELD primary key (HIS_FIELD_ID, HIS_TABLE_ID),
constraint FK_QX_HIS_FIELD_QX_HIS_TAB foreign key (HIS_TABLE_ID) references QX_HIS_TABLE
)
go

create index IDX_QX_HIS_FIELD_TABLE on dbo.QX_HIS_FIELD (HIS_TABLE_ID)
go


create table dbo.QX_HIS_TBLFIELD (
  HIS_TABLE_ID         INT    NOT NULL,
  HIS_FIELD_ID         INT    NOT NULL,
constraint PK_QX_HIS_TBLFIELD primary key (HIS_TABLE_ID, HIS_FIELD_ID),
constraint FK_QX_HIS_TBLFIELD_TABLE    foreign key (HIS_TABLE_ID)            references QX_HIS_TABLE,
constraint FK_QX_HIS_TBLFIELD_FIELD    foreign key (HIS_FIELD_ID, HIS_TABLE_ID)    references QX_HIS_FIELD
)
go

create index IDX_QX_HIS_TBLFIELD_FIELD on dbo.QX_HIS_TBLFIELD (HIS_FIELD_ID)
go


create table dbo.QX_HISTORY (
  qx_historyid int not null,
  qx_history text null,
  qx_object varchar(255) null,
  qx_objectid int null,
  qx_datemodified datetime null
)
go


--
-- TASKS
--

create table dbo.QX_TASK_DELAYUNIT (
  TASK_DELAYUNIT_ID    INT        NOT NULL,
  TASK_DELAYUNIT_NAME    VARCHAR(255)    NOT NULL,
constraint PK_QX_TASK_DELAYUNIT primary key (TASK_DELAYUNIT_ID)
)
go



create table dbo.QX_TASK_STATUS (
  TASK_STATUS_ID    INT        NOT NULL,
  TASK_STATUS_NAME    VARCHAR(255)    NOT NULL,
constraint PK_QX_TASK_STATUS primary key (TASK_STATUS_ID)
)
go


create table dbo.QX_SCRIPT (
  SCRIPT_ID            INT           NOT NULL,
  SCRIPT_NAME          VARCHAR(255)  NOT NULL,
  SCRIPT_DESCRIPTION   VARCHAR(1022) NULL,
  SCRIPT_CLASS_NAME    VARCHAR(255)  NOT NULL,
  CREATOR_ID           INT           NOT NULL,
  DATE_MODIFIED        DATETIME      NOT NULL,
constraint PK_QX_SCRIPT primary key (SCRIPT_ID)
)
go

create unique index IDX_QX_SCRIPT_NAME       on dbo.QX_SCRIPT (SCRIPT_NAME)
create unique index IDX_QX_SCRIPT_CLASS_NAME on dbo.QX_SCRIPT (SCRIPT_CLASS_NAME)
go

create index IDX_QX_SCRIPT_CREATOR      on dbo.QX_SCRIPT (CREATOR_ID)
go



create table dbo.QX_TASK (
  TASK_ID             INT          NOT NULL,
  TASK_NAME           VARCHAR(255) NOT NULL,
  TASK_STATUS_ID      INT          NOT NULL,
  SCRIPT_ID           INT          NOT NULL,
  CREATOR_ID          INT          NOT NULL,
  DATE_CREATED        DATETIME     NOT NULL,
  FIRST_START         DATETIME,
  NEXT_START          DATETIME,
  REPEAT_SIGN         NUMERIC(2)   DEFAULT 0,
  AUTO_START          NUMERIC(2)   DEFAULT 0,
  DELAY               INT,
  TASK_DELAYUNIT_ID   INT,
  LOG_PROCESS_ID      BIGINT,
constraint PK_QX_TASK primary key (TASK_ID),
constraint FK_QX_TASK_QX_TASK_STATUS    foreign key (TASK_STATUS_ID)    references QX_TASK_STATUS (TASK_STATUS_ID),
constraint FK_QX_TASK_QX_TDU        foreign key (TASK_DELAYUNIT_ID)    references QX_TASK_DELAYUNIT (TASK_DELAYUNIT_ID),
constraint FK_QX_TASK_QX_SCRIPT        foreign key (SCRIPT_ID)        references QX_SCRIPT (SCRIPT_ID)
)
go

create index IDX_QX_TASK_CREATOR      on dbo.QX_TASK (CREATOR_ID)
create index IDX_QX_TASK_STATUS       on dbo.QX_TASK (TASK_STATUS_ID)
create index IDX_QX_TASK_DELAYUNIT    on dbo.QX_TASK (TASK_DELAYUNIT_ID)
create index IDX_QX_TASK_SCRIPT       on dbo.QX_TASK (SCRIPT_ID)
go



--
-- LOCKING
--

create table dbo.QX_LOCK (
  PKEY                 INT        NOT NULL,
  TABLE_NAME           VARCHAR(255)    NOT NULL,
  RECORD_ID            VARCHAR(100)    NOT NULL,
  RECORD_ID2           VARCHAR(100)    NOT NULL,
  RECORD_ID3           VARCHAR(100)    NOT NULL,
  RECORD_ID4           VARCHAR(100)    NOT NULL,
  SESSION_ID           VARCHAR(255),
  FOCUS_ID             VARCHAR(255),
  FOCUS_INSTANCE       INT,
  USER_ID              INT,
  USER_TYPE_ID         INT,
  CREATED              DATETIME,
constraint PK_QX_LOCK primary key (PKEY),
constraint FK_QX_LOCK_QX_USER_TYPE foreign key (USER_TYPE_ID) references QX_USER_TYPE (USER_TYPE_ID)
)
go

create unique index IDX_LOCK_TABLE_RECORDS on dbo.QX_LOCK (
  TABLE_NAME,
  RECORD_ID,
  RECORD_ID2,
  RECORD_ID3,
  RECORD_ID4
)
go

create index IDX_QX_LOCK_SESSION   on dbo.QX_LOCK (SESSION_ID)
create index IDX_QX_LOCK_FOCUS     on dbo.QX_LOCK (FOCUS_ID)
create index IDX_QX_LOCK_USER      on dbo.QX_LOCK (USER_ID)
create index IDX_QX_LOCK_USER_TYPE on dbo.QX_LOCK (USER_TYPE_ID)
go

--
-- SEARCH
--

create table dbo.QX_SEARCH (
  SEARCH_ID        INT        NOT NULL,
  NAME            VARCHAR(255)    NOT NULL,
  OWNER_ID        INT,
  TIMESTAMP        DATETIME    NOT NULL,
  FOCUS_ID        VARCHAR(255)    NOT NULL,
  TAB_ID        VARCHAR(255)    NOT NULL,
  FORM_ID        VARCHAR(255)    NOT NULL,
  BODY            TEXT        NULL,
constraint PK_QX_SEARCH primary key (SEARCH_ID)
)
go

create index IDX_QX_SEARCH_OWNER on dbo.QX_SEARCH (OWNER_ID)
create index IDX_QX_SEARCH_FOCUS on dbo.QX_SEARCH (FOCUS_ID)
create index IDX_QX_SEARCH_TAB   on dbo.QX_SEARCH (TAB_ID)
create index IDX_QX_SEARCH_FORM  on dbo.QX_SEARCH (FORM_ID)
go

--
-- MISC
--
create table dbo.QX_STATES (
    pkey              int             not null,
    name              varchar(255)    not null,
    state_id          varchar(2)      not null,
    constraint PK_QX_STATES primary key (pkey)
)
go

create table dbo.QX_MAIL_PROVIDER (
    PKEY int not null,
    PROVIDER_ID varchar(50),
    PROVIDER_NAME varchar(50),

    constraint PK_QX_MAIL_PROVIDER primary key (PKEY),
)
go

create table dbo.QX_AUTO_REPLY (
    PKEY int not null,
    NAME varchar(255) not null,
    DESCRIPTION varchar(255) null,
    SUCCESSFUL_SUBJECT varchar(255) not null,
    SUCCESSFUL_BODY text null,
    UPDATE_SUBJECT varchar(255) not null,
    UPDATE_BODY text null,
    FAIL_SUBJECT varchar(255) not null,
    FAIL_BODY text null,
    CREATED_BY int not null,
    CREATED_DATE datetime,
    REPLY_EMAIL varchar(255) not null,

    constraint PK_QX_AUTO_REPLY primary key (PKEY),
    constraint FK_AUTO_REPLY_CREATED_BY_QX_USERS foreign key (CREATED_BY) references QX_USERS (PKEY),
)ON [PRIMARY]
go

create table dbo.QX_ACCOUNT (
    PKEY int not null,
    OWNER_ID int null,
    WORKGROUP_ID int null,
    ACCOUNT_NAME varchar(255) not null,
    ACCOUNT_PROVIDER int not null,
    EMAIL_SERVER varchar(255) not null,
    EMAIL_SERVER_PORT varchar(50) null,
    EMAIL_SERVER_FOLDER varchar(50) null,
    EMAIL_PASSWORD varchar(255) not null,
    EMAIL_BOX_NAME varchar(255) null,
    LOGIN_NAME varchar(50) not null,
    LASTRECEIVE_DATE datetime,
    MAX_MAIL_SIZE int null,
    MAX_INBOX_SIZE int null,
    DELETE_EMAIL_FLAG int null,
    UNREAD_EMAIL_FLAG int null,
    UNREAD_OUTEMAIL_FLAG int null,
    ACTIVE_ACC int null,
    VALID_ACC int null,
    DELAY int null,
    NEXT_CHECK_DATE datetime,
    NOTES text,
    AUTOREPLY_ID int null,
    CREATED_BY int not null,
    CREATED_DATE datetime,
    REPLY_EMAIL varchar(255) not null,
    LASTUID varchar(512),
    LASTMESSAGE_DIGEST varchar(512),

    constraint PK_QX_ACCOUNT primary key (PKEY),
    constraint FK_ACCOUNT_QX_WORKGROUP foreign key (WORKGROUP_ID) references QX_WORKGROUP (PKEY),
    constraint FK_ACCOUNT_QX_MAIL_PROVIDER foreign key (ACCOUNT_PROVIDER) references QX_MAIL_PROVIDER (PKEY),
    constraint FK_ACCOUNT_CREATED_BY_QX_USERS foreign key (CREATED_BY) references QX_USERS (PKEY),
    constraint FK_ACCOUNT_PROVIDER_ACCOUNT_PROVIDER foreign key (ACCOUNT_PROVIDER) references QX_MAIL_PROVIDER (PKEY),
    constraint FK_ACCOUNT_OWNER_QX_USERS foreign key (OWNER_ID) references QX_USERS (PKEY),
    constraint FK_ACCOUNT_AUTOREPLY_QX_AUTO_REPLY foreign key (AUTOREPLY_ID) references dbo.QX_AUTO_REPLY (PKEY)
)ON [PRIMARY]
go

create table dbo.QX_EMAIL_ROUTING_RULES (
    PKEY int not null,
    OWNER_ID int null,
    WORKGROUP_ID int null,
    RULE_NAME varchar(50) not null,
    KEY_WORD varchar(255) not null,
    FROM_ADDRESS int null,
    TO_ADDRESS int null,
    SUBJECT int null,
    BODY int null,
    AUTOREPLY_ID int null,
    CREATED_BY int not null,
    CREATED_DATE datetime,

    constraint PK_QX_EMAIL_ROUTING_RULES primary key (PKEY),
    constraint FK_ROUTING_QX_WORKGROUP foreign key (WORKGROUP_ID) references QX_WORKGROUP (PKEY),
    constraint FK_ROUTING_CREATED_BY_QX_USERS foreign key (CREATED_BY) references QX_USERS (PKEY),
    constraint FK_ROUTING_OWNER_QX_USERS foreign key (OWNER_ID) references QX_USERS (PKEY)
)
go

create table dbo.QX_EMAIL_ROUTING (
    ACCOUNT_ID int not null,
    ROUTING_RULE int not null,
    constraint PK_QX_EMAIL_ROUTING primary key (ACCOUNT_ID, ROUTING_RULE),
    constraint FK_QX_EMAIL_ROUTING_QX_ACCOUNT foreign key (ACCOUNT_ID) references QX_ACCOUNT(PKEY),
    constraint FK_QX_EMAIL_ROUTING_QX_EMAIL_ROUTING_RULES foreign key (ROUTING_RULE) references QX_EMAIL_ROUTING_RULES(PKEY)
)
go

create table dbo.QX_EMAIL_FILTER (
    PKEY int not null,
    NAME varchar(50) not null,
    FROM_ADDRESS int null,
    TO_ADDRESS int null,
    SUBJECT int null,
    BODY int null,
    WORD varchar(255) not null,
    CREATED_BY int not null,
    CREATED_DATE datetime,

    constraint PK_QX_EMAIL_FILTER primary key (PKEY),
    constraint FK_EMAIL_FILTER_CREATED_BY_QX_USERS foreign key (CREATED_BY) references QX_USERS (PKEY),
)
go

create table dbo.QX_FILTER_ACCOUNT (
    ACCOUNT_ID int not null,
    EMAIL_FILTER int not null,

    constraint PK_QX_FILTER_ACCOUNT primary key (ACCOUNT_ID, EMAIL_FILTER),
    constraint FK_QX_FILTER_ACCOUNT_QX_ACCOUNT foreign key (ACCOUNT_ID) REFERENCES QX_ACCOUNT(PKEY),
    constraint FK_QX_FILTER_ACCOUNT_QX_EMAIL_FILTER foreign key (EMAIL_FILTER) REFERENCES QX_EMAIL_FILTER(PKEY)
)
go

create table dbo.QX_INBOX (
    PKEY int not null,
    MESSAGE_TYPE int not null default 0, -- 0 is email message, 1 - alert
    MESSAGE_SENDER int null,
    MESSAGE_SENDER_EMAIL varchar(1024),
    RECIPIENT_ID int null,
    WORKGROUP_ID int null,
    EMAIL_TO varchar(1024) null,
    EMAIL_CC varchar(1024) null,
    SUBJECT varchar(1024) null,
    MESSAGE text,
    SENT_DATE datetime,
    SERVER_RECEIVED_DATE datetime,
    RECEIVED_DATE datetime,
    OBJECT_TYPE int null,
    OBJECT_ID int null,
    REPLIED_DATE datetime,
    PROCESS_ID int null,
    ACCOUNT_ID int null,
    DIGEST_SRC varchar(512),
    DELETE_FLAG int not null default 0, -- 0 existed, 1 - deleted

    constraint PK_QX_INBOX primary key (PKEY),
    constraint FK_QX_INBOX_SENDER_QX_USERS foreign key (MESSAGE_SENDER) references QX_USERS(PKEY),
    constraint FK_QX_INBOX_RECIPIENT_QX_USERS foreign key (RECIPIENT_ID) references QX_USERS(PKEY),
    constraint FK_QX_INBOX_WORKGROUP_QX_WORKGROUP foreign key (WORKGROUP_ID) references QX_WORKGROUP(PKEY),
    constraint FK_QX_INBOX_OBJECT_TYPE_QX_VIEW_OBJECT_TYPES foreign key (OBJECT_TYPE) references QX_VIEW_OBJECT_TYPES(PKEY),
    constraint FK_QX_INBOX_ACCOUNT_ID_QX_ACCOUNT foreign key (ACCOUNT_ID) references QX_ACCOUNT(PKEY),
)
go

create view dbo.QX_DD_OBJECT_TYPE as
    select
        ENUM_VALUE as PKEY,
        ENUM_LABEL as NAME
    from QX_FIELDINFO
    where (TABLE_NAME = 'qx_inbox') and (COLUMN_NAME = 'object_type')
go

create view qx_user_roles as
    select distinct r.role_id, u.user_id, r.default_focus_id
        from qx_role r, qx_role_to_users u
        where r.role_id = u.role_id
union all
    select r.role_id, g.employeegroup, r.default_focus_id
        from qx_role r, qx_workgroup_to_role wr, qx_groupmember g
        where r.role_id = wr.role_id
        and wr.workgroup_id = g.workgroupgroup
go

create table dbo.QX_EMAIL_SERVICE_LOG (
    PKEY int NOT NULL ,
    MESSAGE_ID int NULL ,
    MESSAGE_TYPE int NULL ,
    E_TO varchar(255) NULL ,
    E_FROM varchar(255) NULL ,
    E_SUBJECT varchar(1022) NULL ,
    MESSAGE_LEVEL int NULL ,
    CREATED datetime NULL ,
    OWNER_ID int NULL ,
    WORKGROUP_ID int NULL ,
    OBJECT_TYPE int null,
    OBJECT_ID int null,
    FILTER_ID int NULL ,
    ROUTING_ID int NULL ,
    DESCRIPTION varchar(1022) NULL ,
    ACCOUNT_ID int NULL ,
    constraint PK_QX_EMAIL_SERVICE_LOG primary key (PKEY)
)
go

create table dbo.QX_HELPLINK (
  TAB_ID varchar(255),
  LINK varchar(255),
  constraint PK_QX_HELPLINK primary key (TAB_ID)
)
go

create table dbo.QX_PORTAL_CONFIG (
  NAME varchar(255) not null,
  CSS_FILE varchar(1024) null,
  SHOW_FOCUS_MENU bit default 1,
  SHOW_SUBFOCUS_MENU bit default 1,
  constraint PK_QX_PORTAL_CONFIG primary key(NAME)
)
go

create table dbo.QX_PORTAL_TOOLBAR (
  PORTAL_CONFIG_NAME varchar(255) not null,
  ORDER_ID int not null,
  CONTROL_NAME varchar(255) not null,
  constraint PK_QX_PORTAL_TOOLBAR primary key(PORTAL_CONFIG_NAME, ORDER_ID),
  constraint FK_QX_PORTAL_TOOLBAR_PORTAL_CONFIG foreign key (PORTAL_CONFIG_NAME) references QX_PORTAL_CONFIG(NAME)
)
go
