-- ------------ --
-- Domain model --
-- ------------ --

--
-- SYSTEM tables
--

create table QX_KEYS (
  TABLENAME        VARCHAR2(240)   NOT NULL,
  KEYVALUE         INT             NOT NULL,
constraint PK_QX_KEYS primary key (TABLENAME)
);



create table QX_FIELDINFO (
  TABLE_NAME        VARCHAR2(18)   NOT NULL,
  COLUMN_NAME       VARCHAR2(18)   NOT NULL,
  COLUMN_TYPE       NUMBER(2)     NOT NULL,
  ENUM_VALUE        NUMBER(2),
  ENUM_LABEL        VARCHAR2(128)
);

create unique index IDX_QX_FIELDINFO on QX_FIELDINFO (
  TABLE_NAME,
  COLUMN_NAME,
  ENUM_VALUE
);



create table QX_SYS_PROP (
  PKEY            INT            NOT NULL,
  PROP_ID         VARCHAR2(255)  NOT NULL,
  PROP_VAL        VARCHAR2(255),
  PROP_TEXT       CLOB,
  PROP_DATA       BLOB,
  NEED_RESTART    NUMBER(1)        DEFAULT 0,
  DESCRIPTION     VARCHAR2(255),
constraint PK_QX_SYS_PROP primary key (PKEY)
);



create table QX_LANGUAGE (
  PKEY          INT           NOT NULL,
  LANGUAGE_ID   CHAR(2)       NOT NULL,
  NAME          VARCHAR2(255) NOT NULL,
constraint PK_QX_LANGUAGE primary key (LANGUAGE_ID)
);



create table QX_TIMEZONE (
  PKEY           INT           NOT NULL,
  TIMEZONE_ID    VARCHAR2(255) NOT NULL,
  NAME           VARCHAR2(255) NOT NULL,
  OFFSET         INT           NOT NULL,
constraint PK_QX_TIMEZONE primary key (PKEY)
);



create table QX_COUNTRY (
  PKEY                 INT           NOT NULL,
  COUNTRY_ID           VARCHAR2(3)   NOT NULL,
  NAME                 VARCHAR2(255) NOT NULL,
  DEF_LANG_ID          CHAR(2),
  DEF_TIMEZONE_ID      VARCHAR2(255),
constraint PK_QX_COUNTRY primary key (PKEY)
);

create index IDX_QX_COUNTRY_LANGUAGE on QX_COUNTRY (DEF_LANG_ID);
create index IDX_QX_COUNTRY_TIMEZONE on QX_COUNTRY (DEF_TIMEZONE_ID);


CREATE TABLE QX_TIME_PATTERN (
  PKEY             INT           NOT NULL,
  TIME_PATTERN_ID  VARCHAR2(255) NOT NULL,
constraint PK_QX_TIME_PATTERN primary key (PKEY)
);

CREATE TABLE QX_DATE_PATTERN (
  PKEY             INT           NOT NULL,
  DATE_PATTERN_ID  VARCHAR2(255) NOT NULL,
constraint PK_QX_DATE_PATTERN primary key (PKEY)
);

create view QX_DD_DATEPOS as
    select
        ENUM_VALUE as PKEY,
        ENUM_LABEL as NAME
    from QX_FIELDINFO
    where (TABLE_NAME = 'qx_user_settings') and (COLUMN_NAME = 'date_position')
;

create view QX_DD_YESNO as
    select
        ENUM_VALUE as PKEY,
        ENUM_LABEL as NAME
    from QX_FIELDINFO
    where (TABLE_NAME = '_FAKE_YES_NO') and (COLUMN_NAME = 'FAKE_COLUMN')
;

--
-- ENTITY/FOCUS
--

create table QX_ENTITY (
  ENTITY_ID            VARCHAR2(255)    NOT NULL,
  ENTITYDATA           BLOB,
constraint PK_QX_ENTITY primary key (ENTITY_ID)
);

--
-- SECURITY
--

create table QX_VIEW_OBJECT_TYPES (
    pkey                int             not null,
    type_description    VARCHAR2(255)   not null,
constraint PK_QX_VIEW_OBJECT_TYPES primary key (pkey)
);

create table QX_VIEW_OBJECTS (
    pkey                        int             not null,
    name                        VARCHAR2(255)   not null,
    type                        int             not null,
    parent_object               int             null,
    order_in_group              int             default 0 not null,
    icon                        VARCHAR2(255)   null,
    inframelinks				number(1)       default 0 not null,
    grid				        number(1)       default 1 not null,
constraint PK_QX_VIEW_OBJECTS primary key (pkey),
constraint FK_QX_VIEW_OBJ_TO_OBJ_TYPES foreign key (type) references QX_VIEW_OBJECT_TYPES(pkey),
constraint FK_QX_VIEW_OBJ_TO_PARENT_OBJ foreign key (parent_object) references QX_VIEW_OBJECTS(pkey)
);

create table QX_FORM_SETTINGS (
    view_object_id          int         not null,
    entity_id               VARCHAR2(255),
    grid                    int,
    use_default_actions     number(1)         default 1,
    labels_layout           number(1)         default 0,
    myqueweb           	    number(1)         default 1 not null,
    autosearch           	number(1)         default 0 not null,
constraint FK_QX_FORM_SET_TO_VIEW_OBJ foreign key (view_object_id) references QX_VIEW_OBJECTS(pkey)
);

create table QX_ROLE (
    role_id           int             not null,
    default_focus_id  int             null,
    name              VARCHAR2(255)   not null,
    description       VARCHAR2(255)   null,
constraint PK_QX_ROLE primary key (role_id),
constraint FK_QX_ROLE_TO_FOCUS foreign key (default_focus_id) references QX_VIEW_OBJECTS(pkey)
);

create table QX_ACCESSLEVEL (
    pkey    int             not null,
    name    VARCHAR2(255)   not null,
constraint PK_QX_ACCESSLEVEL primary key (pkey)
);

create table QX_PERMISSION (
    permission_id           int              not null,
    role_id                 int              not null,
    access_level            int              not null,
    permission_object_type  int              not null,
    object_id               int              not null,
constraint PK_QX_PERMISSION primary key (permission_id),
constraint FK_QX_PERM_TO_ACCESS_LVL foreign key (access_level) references qx_accesslevel(pkey),
constraint FK_QX_PERM_TO_ROLE foreign key (role_id) references qx_role(role_id),
constraint FK_QX_PERM_TO_PERM_OBJ_TYPES foreign key (permission_object_type) references QX_VIEW_OBJECT_TYPES(pkey),
constraint FK_QX_PERM_TO_QX_VIEW_OBJ foreign key (object_id) references QX_VIEW_OBJECTS(pkey)
);

create table QX_USER_TYPE (
    USER_TYPE_ID         INT            NOT NULL,
    NAME                 VARCHAR2(255)  NOT NULL,
constraint PK_QX_USER_TYPE primary key (USER_TYPE_ID)
);

create table QX_USERS (
    PKEY                    INT             NOT NULL,
    LOGINNAME               VARCHAR2(255)   NOT NULL,
    FULLNAME                VARCHAR2(255)   NULL,
    PASSWORD                VARCHAR2(255)   NULL,
    EMAIL                   VARCHAR2(255)   NULL,
    USER_TYPE               INT             NOT NULL,
constraint PK_QX_USERS primary key (PKEY),
constraint FK_QX_USERS_TO_USERTYPE foreign key (USER_TYPE) references qx_user_type(USER_TYPE_ID)
);

create unique index IDX_USERS_LOGINNAME on QX_USERS (LOGINNAME);


create table QX_WORKGROUP (
    pkey                int             not null,
    name                VARCHAR2(40)    not null,
    description         VARCHAR2(80),
    notificationaddr    VARCHAR2(40),
    notifymethod        NUMBER(2),
constraint PK_QX_WORKGROUP primary key (PKEY)
);

create table QX_GROUPMEMBER (
  PKEY              INT     NOT NULL,
  EMPLOYEEGROUP     INT     NOT NULL,
  WORKGROUPGROUP    INT     NOT NULL,
  NOTIFYMETHOD      NUMBER(2),
  TIER              INT     DEFAULT 0,
constraint PK_QX_GROUPMEMBER primary key (PKEY),
constraint FK_QX_GROUPMEMBER_EMPLOYEE   foreign key (EMPLOYEEGROUP)     references QX_USERS(PKEY),
constraint FK_QX_GROUPMEMBER_WORKGROUP  foreign key (WORKGROUPGROUP)    references QX_WORKGROUP(PKEY)
);

create index IDX_QX_GROUPMEMBER_EMPLOYEE  on QX_GROUPMEMBER (EMPLOYEEGROUP);
create index IDX_QX_GROUPMEMBER_WORKGROUP on QX_GROUPMEMBER (WORKGROUPGROUP);


create table QX_ROLE_TO_USERS (
    pkey                int             not null,
    user_id             int             not null,
    role_id             int             not null,
constraint FK_QX_ROLE_TO_USER_TO_ROLE_FK foreign key (role_id) references QX_ROLE(role_id),
constraint FK_QX_ROLE_TO_USER_TO_USER_FK foreign key (user_id) references QX_USERS(pkey)
);

create table QX_WORKGROUP_TO_ROLE (
    pkey                int             not null,
    role_id             int             not null,
    workgroup_id        int             not null,
constraint FK_QX_WGROUP_ROLE_TO_WGROUP_FK foreign key (workgroup_id) references QX_WORKGROUP(pkey),
constraint FK_QX_WGROUP_ROLE_TO_ROLE_FK foreign key (role_id) references QX_ROLE(role_id)
);

create view qx_user_permissions as
    select u.pkey as user_id, p.*, vo.name as object_name
    from qx_permission p
		inner join qx_role_to_users r2u on p.role_id = r2u.role_id
		inner join qx_users u on u.pkey = r2u.user_id
		inner join qx_view_objects vo on p.object_id = vo.pkey
    union
    select u.pkey as user_id, p.*, vo.name as object_name
    from qx_permission p
		inner join qx_workgroup_to_role w2r on p.role_id = w2r.role_id
		inner join qx_workgroup w on w.pkey = w2r.workgroup_id
		inner join qx_groupmember gm on gm.workgroupgroup = w.pkey
		inner join qx_users u on u.pkey = gm.employeegroup
		inner join qx_view_objects vo on p.object_id = vo.pkey
;

--
-- USERS, WORKGROUPS, ROLES
--
create view QX_DD_WORKGROUPNOTIFYMETHOD as
    select
        enum_value as PKEY,
        enum_label as NAME
    from qx_fieldinfo
    where (table_name = 'qx_workgroup') and (column_name = 'notifymethod')
;

create view QX_DD_GROUPMEMBERNOTIFYMETHOD as
    select
        ENUM_VALUE as PKEY,
        ENUM_LABEL as NAME
    from QX_FIELDINFO
    where (TABLE_NAME = 'qx_groupmember') and (COLUMN_NAME = 'notifymethod')
;

create table QX_USER_PROP (
  PROP_ID          VARCHAR2(512) NOT NULL,
  USER_ID          INT           NOT NULL,
  USER_TYPE_ID     INT           NOT NULL,
  PROP_VAL         VARCHAR2(2048),
constraint PK_QX_USER_PROP primary key (PROP_ID, USER_ID, USER_TYPE_ID),
constraint FK_QX_USER_PROP_QX_USER foreign key (USER_TYPE_ID) references QX_USER_TYPE (USER_TYPE_ID)
);

create index IDX_QX_USER_PROP_USER      on QX_USER_PROP (USER_ID);
create index IDX_QX_USER_PROP_USER_TYPE on QX_USER_PROP (USER_TYPE_ID);


create table QX_USER_SETTINGS (
  PKEY        INT  NOT NULL,
  USER_ID     INT  NOT NULL,
  LANG        INT  NULL,
  COUNTRY     INT  NULL,
  TIMEZONE    INT  NULL,
  DPATTERN    INT  NULL,
  TPATTERN    INT  NULL,
  DPOS        INT  NULL,
constraint PK_USER_SETTINGS primary key (PKEY)
);

create unique index IDX_USER_SETTINGS_USER_ID on QX_USER_SETTINGS (USER_ID);


--
-- View
--
create table QX_HEADER (
  FORM_ID              VARCHAR2(255) NOT NULL,
  HEADER_ORDER         INT NOT NULL,
  CLIENT_WIDTH         INT,
constraint PK_QX_HEADER primary key (FORM_ID, HEADER_ORDER)
);

create table QX_ROW (
  FORM_ID              VARCHAR2(255) NOT NULL,
  ROW_ORDER            INT NOT NULL,
constraint PK_QX_ROW primary key (FORM_ID, ROW_ORDER)
);

create table QX_COL (
  ROW_ORDER            VARCHAR2(255) NOT NULL,
  FORM_ID              VARCHAR2(255) NOT NULL,
  COL_ORDER            INT NOT NULL,
  FIELD_ID             VARCHAR2(255),
  ROWSPAN              INT,
  COLSPAN              INT,
constraint PK_QX_COL primary key (ROW_ORDER, FORM_ID, COL_ORDER)
);

create table QX_BUTTON (
  FORM_ID              VARCHAR2(255) NOT NULL,
  BUTTON_ID            VARCHAR2(255) NOT NULL,
  BUTTON_ORDER         INT,
  PERMISSION           INT,
constraint PK_QX_BUTTON primary key (FORM_ID, BUTTON_ID)
);

create table QX_HTML_ELEMENT (
  FORM_ID              VARCHAR2(255) NOT NULL,
  HTML_ELEMENT_ID      VARCHAR2(255) NOT NULL,
  HTML_ELEMENT_ORDER   INT,
constraint PK_QX_HTML_ELEMENT primary key (FORM_ID, HTML_ELEMENT_ID)
);

create table QX_HIDDENCONTROL (
  FORM_ID              VARCHAR2(255) NOT NULL,
  FIELD_ID            VARCHAR2(255) NOT NULL,
constraint PK_QX_HIDDENCONTROL primary key (FORM_ID, FIELD_ID)
);

create table QX_LINK (
  FORM_ID              VARCHAR2(255) NOT NULL,
  FIELD_ID             VARCHAR2(255) NOT NULL,
  LINKED_FORM_ID       VARCHAR2(255) NOT NULL,
constraint PK_QX_LINK primary key (FORM_ID, FIELD_ID)
);

create table QX_EXTERNALFIELD (
  EXTERNALFORM_ID      VARCHAR2(255)    NOT NULL,
  FORM_ID              VARCHAR2(255)    NOT NULL,
  EXTERNALFIELD        VARCHAR2(255),
  SOURCEFIELD          VARCHAR2(255),
constraint PK_QX_EXTERNALFIELD primary key (EXTERNALFORM_ID, FORM_ID)
);

create index IDX_QX_EXTERNALFIELD_FORM  on QX_EXTERNALFIELD (FORM_ID);
create index IDX_QX_EXTERNALFIELD_FIELD on QX_EXTERNALFIELD (EXTERNALFIELD);


create table QX_EXTERNALFORM (
  EXTERNALFORM_ID      VARCHAR2(255)    NOT NULL,
  FORM_ID              VARCHAR2(255)    NOT NULL,
constraint PK_QX_EXTERNALFORM primary key (EXTERNALFORM_ID, FORM_ID)
);

create index IDX_QX_EXTERNALFORM_FORM on QX_EXTERNALFORM (FORM_ID);


create table QX_EXTERNALSET (
  EXTERNALFORM_ID      VARCHAR2(255)    NOT NULL,
  FORM_ID              VARCHAR2(255)    NOT NULL,
constraint PK_QX_EXTERNALSET primary key (EXTERNALFORM_ID, FORM_ID)
);

create index IDX_QX_EXTERNALSET_FORM on QX_EXTERNALSET (FORM_ID);



create table QX_LINKEDDATASET (
  ENTITY_ID            VARCHAR2(255)    NOT NULL,
  DATASET_ID           VARCHAR2(255)    NOT NULL,
  FORM_ID              VARCHAR2(255)    NOT NULL,
constraint PK_QX_LINKEDDATASET primary key (ENTITY_ID, DATASET_ID)
);

create index IDX_QX_LINKEDDATASET_FORM on QX_LINKEDDATASET (FORM_ID);


--
-- Localization
--

create table QX_LOCALIZATION_TYPE (
  LOCALIZATION_TYPE_ID   INT             NOT NULL,
  NAME                   VARCHAR2(255)   NOT NULL,
constraint PK_QX_LOCALIZATION_TYPE primary key (LOCALIZATION_TYPE_ID)
);

create table QX_LOCALIZED_OBJECT_TYPE (
  OBJECT_TYPE_ID  INT               NOT NULL,
  NAME            VARCHAR2(255)     NOT NULL,
constraint PK_QX_LOCALIZED_OBJECT_TYPE primary key (OBJECT_TYPE_ID)
);

create table QX_LOCALIZATION (
  LOCALIZATION_ID	    VARCHAR2(255)     NOT NULL,
  LOCALIZATION_TYPE_ID  INT               NOT NULL,
  OBJECT_TYPE_ID	    INT               NOT NULL,
  LANGUAGE_ID		    CHAR(2)           NOT NULL,
  CONTENT               NVARCHAR2(2000)   NULL,
constraint PK_QX_LOCALIZATION primary key (LOCALIZATION_ID, LOCALIZATION_TYPE_ID, LANGUAGE_ID, OBJECT_TYPE_ID),
constraint FK_QX_LOCAL_QX_LANGUAGE foreign key (LANGUAGE_ID) references QX_LANGUAGE (LANGUAGE_ID),
constraint FK_QX_LOCAL_TYPE foreign key (LOCALIZATION_TYPE_ID) references QX_LOCALIZATION_TYPE (LOCALIZATION_TYPE_ID),
constraint FK_QX_LOCAL_OBJECT_TYPE foreign key (OBJECT_TYPE_ID) references QX_LOCALIZED_OBJECT_TYPE (OBJECT_TYPE_ID)
);

create index IDX_QX_LOCALIZATION_LANGUAGE on QX_LOCALIZATION (LANGUAGE_ID);
create index IDX_QX_LOCALIZATION_OBJECT on QX_LOCALIZATION (OBJECT_TYPE_ID);



--
-- Context menu
--

create table QX_CONTEXTMENU (
  CONTEXTMENU_ID    VARCHAR2(255)    NOT NULL,
constraint PK_QX_CONTETXMENU primary key (CONTEXTMENU_ID)
);

create table QX_MENUITEM (
  MENUITEM_ID       VARCHAR2(255)   NOT NULL,
  CONTEXTMENU_ID    VARCHAR2(255)   NOT NULL,
  MENUITEM_ORDER    INT,
constraint PK_QX_MENUITEM primary key (MENUITEM_ID)
);

create index IDX_QX_MENUITEM_CONTEXTMENU on QX_MENUITEM (CONTEXTMENU_ID);
create index IDX_QX_MENUITEM_ORDER on QX_MENUITEM (MENUITEM_ORDER);


create table QX_CONTEXTMENU_FORM (
  FORM_ID           VARCHAR2(255)    NOT NULL,
  CONTEXTMENU_ID    VARCHAR2(255)    NOT NULL,
constraint PK_QX_CONTETXMENU_FORM primary key (FORM_ID, CONTEXTMENU_ID)
);

--
-- ALERTS
--

create table QX_ALERT (
  ALERT_ID             INT        NOT NULL,
  SENDER_ID            INT        NOT NULL,
  SENDER_TYPE          INT        NOT NULL,
  RECIPIENT_ID         INT,
  RECIPIENT_TYPE       INT,
  WORKGROUP_ID         INT,
  TIER                 NUMBER(2),
  TO_ALL               NUMBER(2)               DEFAULT 0,
  FOCUS_ID             VARCHAR2(255),
  TAB_ID               VARCHAR2(255),
  FORM_ID              VARCHAR2(255),
  RECORD_ID            VARCHAR2(100),
  RECORD_ID2           VARCHAR2(100),
  RECORD_ID3           VARCHAR2(100),
  RECORD_ID4           VARCHAR2(100),
  MESSAGE              VARCHAR2(1022)    NOT NULL,
  SEVERITY             NUMBER(2)    NOT NULL,
  DATEPOSTED           DATE        NOT NULL,
  CREATOR_ID           INT        NOT NULL,
  CREATOR_TYPE         INT        NOT NULL,
constraint PK_QX_ALERT primary key (ALERT_ID),
constraint FK_QX_ALERT_QX_USER_TYPE1    foreign key (SENDER_TYPE)    references QX_USER_TYPE (USER_TYPE_ID),
constraint FK_QX_ALERT_QX_USER_TYPE2    foreign key (RECIPIENT_TYPE)    references QX_USER_TYPE (USER_TYPE_ID),
constraint FK_QX_ALERT_QX_USER_TYPE3    foreign key (CREATOR_TYPE)    references QX_USER_TYPE (USER_TYPE_ID)
);

create index IDX_QX_ALERT_SENDER         on QX_ALERT (SENDER_ID);
create index IDX_QX_ALERT_SENDER_TYPE    on QX_ALERT (SENDER_TYPE);
create index IDX_QX_ALERT_RECIPIENT      on QX_ALERT (RECIPIENT_ID);
create index IDX_QX_ALERT_RECIPIENT_TYPE on QX_ALERT (RECIPIENT_TYPE);
create index IDX_QX_ALERT_CREATOR        on QX_ALERT (CREATOR_ID);
create index IDX_QX_ALERT_CREATOR_TYPE   on QX_ALERT (CREATOR_TYPE);
create index IDX_QX_ALERT_WORKGROUP      on QX_ALERT (WORKGROUP_ID);
create index IDX_QX_ALERT_FOCUS          on QX_ALERT (FOCUS_ID);
create index IDX_QX_ALERT_TAB            on QX_ALERT (TAB_ID);
create index IDX_QX_ALERT_FORM           on QX_ALERT (FORM_ID);



create table QX_ALERT_BLOCK (
  ALERT_BLOCK_ID       INT        NOT NULL,
  ALERT_ID             INT        NOT NULL,
  RECIPIENT_ID         INT        NOT NULL,
  RECIPIENT_TYPE       INT        NOT NULL,
constraint PK_QX_ALERT_BLOCK primary key (ALERT_BLOCK_ID),
constraint FK_QX_ALERT_BLOCK_QX_ALERT        foreign key (ALERT_ID)        references QX_ALERT (ALERT_ID),
constraint FK_QX_ALERT_BLOCK_QX_USER_TYPE    foreign key (RECIPIENT_TYPE)    references QX_USER_TYPE (USER_TYPE_ID)
);

create unique index IDX_ALERT_BLOCK on QX_ALERT_BLOCK (
  ALERT_ID,
  RECIPIENT_ID,
  RECIPIENT_TYPE
);

create index IDX_QX_ALERT_BLOCK_REC      on QX_ALERT_BLOCK (RECIPIENT_ID);
create index IDX_QX_ALERT_BLOCK_REC_TYPE on QX_ALERT_BLOCK (RECIPIENT_TYPE);


create view QX_DD_ALERTSEVERITY as
    select
        ENUM_VALUE as PKEY,
        ENUM_LABEL as NAME
    from QX_FIELDINFO
    where (TABLE_NAME = 'qx_alert') and (COLUMN_NAME = 'severity')
;


--
-- ATTACHMENTS
--

create table QX_ATTACHMENT (
  ATTACHMENT_ID  INT        NOT NULL,
  PROCESS_ID     NUMBER(38),
  FILENAME       VARCHAR2(255) NOT NULL,
  CREATED        DATE    NOT NULL,
  INDEXED        CHAR(1),
  DESCRIPTION    VARCHAR2(250),
  FILETYPE       VARCHAR2(50)    DEFAULT 'application/octet-stream',
  FILEEXT        VARCHAR2(10),
  DATA           BLOB,
  USER_ID        INT,
constraint PK_QX_ATTACHMENT primary key (ATTACHMENT_ID)
);

create index IDX_QX_ATTACHMENT_USER      on QX_ATTACHMENT (USER_ID);



create table QX_TEMP_ATTACH (
  TEMP_ATTACH_ID  INT        NOT NULL,
  PROCESS_ID      NUMBER(38),
  FILENAME        VARCHAR2(255),
  CREATED         DATE    NOT NULL,
  DATA            BLOB,
  FILETYPE        VARCHAR2(50)               DEFAULT 'application/octet-stream',
  FILEEXT         VARCHAR2(10),
  USER_TYPE_ID    INT,
  USER_ID         INT,
constraint PK_QX_TEMP_ATTACH primary key (TEMP_ATTACH_ID),
constraint FK_QX_TEMP_ATTACH_QX_USER_TYPE foreign key (USER_TYPE_ID) references QX_USER_TYPE (USER_TYPE_ID)
);

create index IDX_QX_TEMP_ATTACH_USER_TYPE on QX_TEMP_ATTACH (USER_TYPE_ID);
create index IDX_QX_TEMP_ATTACH_USER      on QX_TEMP_ATTACH (USER_ID);

create table QX_ATTACHMENT_OBJECTS (
  PKEY             INT NOT NULL,
  ATTACHMENT_ID    INT,
  OBJECT_ID        INT,
  OBJECT_TYPE      INT,
constraint PK_QX_ATTACHMENT_OBJECTS primary key (PKEY),
constraint FK_QX_ATTACHMENT_OBJ_ATTACH foreign key (ATTACHMENT_ID) references QX_ATTACHMENT (ATTACHMENT_ID)
);


--
-- IMAGES
--

create table QX_IMAGE (
  IMAGE_ID             INT        NOT NULL,
  FILE_SIZE            INT,
  CREATED              DATE        NOT NULL,
  MODIFIER_ID          INT        NOT NULL,
  MODIFIED             DATE        NOT NULL,
  FILENAME             VARCHAR2(255)    NOT NULL,
  FOLDER               VARCHAR2(255),
  CREATOR_ID           INT        NOT NULL,
constraint PK_QX_IMAGE primary key (IMAGE_ID)
);

create index IDX_QX_IMAGE_MODIFIER      on QX_IMAGE (MODIFIER_ID);
create index IDX_QX_IMAGE_CREATOR       on QX_IMAGE (CREATOR_ID);



create table QX_IMAGE_FOLDER (
  FOLDER_ID            INT        NOT NULL,
  FOLDER               VARCHAR2(255)    NOT NULL,
constraint PK_QX_IMAGE_FOLDER primary key (FOLDER_ID)
);



create table QX_IMAGE_USAGE (
  IMAGE_USAGE_ID       INT        NOT NULL,
  IMAGE_ID             INT        NOT NULL,
  RECORD_KEY           INT        NOT NULL,
  TABLE_NAME           VARCHAR2(50)    NOT NULL,
constraint PK_QX_IMAGE_USAGE primary key (IMAGE_USAGE_ID),
constraint FK_QX_IMAGE_USAGE_QX_IMAGE foreign key (IMAGE_ID) references QX_IMAGE (IMAGE_ID)
);

create index IDX_QX_IMAGE_USAGE_IMAGE on QX_IMAGE_USAGE (IMAGE_ID);



--
-- LOGGING
--

create table QX_LOG_TYPE (
  LOG_TYPE_ID       INT          NOT NULL,
  NAME              VARCHAR2(50) NOT NULL,
constraint PK_QX_LOG_TYPE primary key (LOG_TYPE_ID)
);



create table QX_LOG_CLASS (
  LOG_CLASS_ID         INT        NOT NULL,
  NAME                 VARCHAR2(50)    NOT NULL,
  TABLE_NAME           VARCHAR2(50)    NOT NULL,
constraint PK_QX_LOG_CLASS primary key (LOG_CLASS_ID)
);



create table QX_LOG (
  LOG_ID               INT        NOT NULL,
  LOG_PROCESS_ID       NUMBER(38)     NOT NULL,
  LOG_TYPE_ID          INT        NOT NULL,
  LOG_CLASS_ID         INT        NOT NULL,
  USER_TYPE_ID         INT        NOT NULL,
  USER_ID              INT        NOT NULL,
  DESCRIPTION          VARCHAR2(1022),
  STAMP                DATE        NOT NULL,
constraint PK_QX_LOG primary key (LOG_ID),
constraint FK_QX_LOG_QX_USER_TYPE   foreign key (USER_TYPE_ID)    references QX_USER_TYPE (USER_TYPE_ID),
constraint FK_QX_LOG_QX_LOG_TYPE    foreign key (LOG_TYPE_ID)    references QX_LOG_TYPE (LOG_TYPE_ID),
constraint FK_QX_LOG_QX_LOG_CLASS   foreign key (LOG_CLASS_ID)    references QX_LOG_CLASS (LOG_CLASS_ID)
);

create index IDX_QX_LOG_USER_TYPE on QX_LOG (USER_TYPE_ID);
create index IDX_QX_LOG_USER      on QX_LOG (USER_ID);
create index IDX_QX_LOG_LOG_TYPE  on QX_LOG (LOG_TYPE_ID);
create index IDX_QX_LOG_LOG_CLASS on QX_LOG (LOG_CLASS_ID);



create table QX_ERROR_CFG (
  ERROR_CFG_ID       INT        NOT NULL,
  TARGET_CLASS_MASK  VARCHAR2(255)    NOT NULL,
  MESSAGE_HND_CLASS  VARCHAR2(255)    NOT NULL,
  EMAILS             VARCHAR2(1024),
  EMPLOYEE_ID        INT,
  WORKGROUP_ID       INT,
  DISABLE            NUMBER(2)               DEFAULT 0,
constraint PK_QX_ERROR_CFG primary key (ERROR_CFG_ID)
);



create table QX_USER_LOG (
  USER_LOG_ID          INT        NOT NULL,
  USER_ID              INT        NOT NULL,
  USER_TYPE_ID         INT        NOT NULL,
  SESSION_ID           VARCHAR2(255)    NOT NULL,
  FULLNAME             VARCHAR2(255),
  HOST                 VARCHAR2(255),
  REMOTE_ADDRESS       VARCHAR2(15),
  LOGIN_TIME           DATE        NOT NULL,
  LOGOUT_TIME          DATE,
constraint PK_QX_USER_LOG primary key (USER_LOG_ID),
constraint FK_QX_USER_LOG_QX_USER_TYPE foreign key (USER_TYPE_ID) references QX_USER_TYPE (USER_TYPE_ID)
);

create index IDX_QX_USER_LOG_USER      on QX_USER_LOG (USER_ID);
create index IDX_QX_USER_LOG_USER_TYPE on QX_USER_LOG (USER_TYPE_ID);

--
-- REPORTS
--

create table QX_REPORT (
    REPORT_ID       INT        NOT NULL,
    NAME            VARCHAR2(255)    NOT NULL,
    OWNER_ID        INT,
    PUBLIC_REPORT   NUMBER(2)               DEFAULT 0,
    TIMESTAMP       DATE    NOT NULL,
    BODY            CLOB        NOT NULL,
constraint PK_QX_REPORT primary key (REPORT_ID)
);

create index IDX_QX_REPORT_OWNER         on QX_REPORT (OWNER_ID);
create index IDX_QX_REPORT_PUBLIC_REPORT on QX_REPORT (PUBLIC_REPORT);


create table QX_REPORT_SCHED (
  REPORT_SCHED_ID  INT        NOT NULL,
  REPORT_ID        INT        NOT NULL,
  PERIOD           INT        NOT NULL,
  PERIOD_UNIT      INT        NOT NULL,
  ACTIVE           NUMBER(2)               DEFAULT 1,
  CREATED_BY       INT,
  CREATOR_TYPE     INT,
  NEXT_DATE        DATE,
  TO_ADDR          VARCHAR2(255)    NOT NULL,
  CC_ADDR          VARCHAR2(255),
  SUBJECT          VARCHAR2(255)    NOT NULL,
constraint PK_QX_REPORT_SCHED primary key (REPORT_SCHED_ID),
constraint FK_QX_REPSCHED_QX_USER_TYPE foreign key (CREATOR_TYPE) references QX_USER_TYPE (USER_TYPE_ID)
);

create index IDX_REPORT_SCHED on QX_REPORT_SCHED (
  NEXT_DATE,
  ACTIVE
);

create index IDX_QX_RPT_SCHED_REPORT       on QX_REPORT_SCHED (REPORT_ID);
create index IDX_QX_RPT_SCHED_CREATED_BY   on QX_REPORT_SCHED (CREATED_BY);
create index IDX_QX_RPT_SCHED_CREATOR_TYPE on QX_REPORT_SCHED (CREATOR_TYPE);



create view QX_DD_REPORT_SCHED_PERIODUNITS AS
    select
        ENUM_VALUE as PKEY,
        ENUM_LABEL as NAME
    from QX_FIELDINFO
    where (TABLE_NAME = 'qx_report_sched') and (COLUMN_NAME = 'period_unit')
;


--
-- CHARTS
--

create table QX_CHART (
    CHART_ID        INT             NOT NULL,
    NAME            VARCHAR2(255)   NOT NULL,
    OWNER_ID        INT,
    PUBLIC_CHART    NUMBER(2)       DEFAULT 0 NOT NULL,
    SYSTEM_CHART    NUMBER(2)       DEFAULT 0 NOT NULL,
    TIMESTAMP       DATE            NOT NULL,
    BODY            CLOB            NOT NULL,
    constraint PK_QX_CHART primary key (CHART_ID)
);

create index IDX_QX_CHART_OWNER        on QX_CHART (OWNER_ID);
create index IDX_QX_CHART_PUBLIC_CHART on QX_CHART (PUBLIC_CHART);
create index IDX_QX_CHART_SYSTEM_CHART on QX_CHART (SYSTEM_CHART);


--
-- HISTORY
--

create table QX_HIS_TABLE (
  HIS_TABLE_ID         INT        NOT NULL,
  NAME                 VARCHAR2(255)    NOT NULL,
constraint PK_QX_HIS_TABLE primary key (HIS_TABLE_ID)
);


create table QX_HIS_FIELD (
  HIS_FIELD_ID         INT        NOT NULL,
  HIS_TABLE_ID         INT        NOT NULL,
  NAME                 VARCHAR2(255)    NOT NULL,
constraint PK_QX_HIS_FIELD primary key (HIS_FIELD_ID, HIS_TABLE_ID),
constraint FK_QX_HIS_FIELD_QX_HIS_TAB foreign key (HIS_TABLE_ID) references QX_HIS_TABLE
);

create index IDX_QX_HIS_FIELD_TABLE on QX_HIS_FIELD (HIS_TABLE_ID);


create table QX_HIS_TBLFIELD (
  HIS_TABLE_ID         INT    NOT NULL,
  HIS_FIELD_ID         INT    NOT NULL,
constraint PK_QX_HIS_TBLFIELD primary key (HIS_TABLE_ID, HIS_FIELD_ID),
constraint FK_QX_HIS_TBLFIELD_TABLE    foreign key (HIS_TABLE_ID)            references QX_HIS_TABLE,
constraint FK_QX_HIS_TBLFIELD_FIELD    foreign key (HIS_FIELD_ID, HIS_TABLE_ID)    references QX_HIS_FIELD
);

create index IDX_QX_HIS_TBLFIELD_FIELD on QX_HIS_TBLFIELD (HIS_FIELD_ID);


create table QX_HISTORY (
  qx_historyid int not null,
  qx_history CLOB null,
  qx_object VARCHAR2(255) null,
  qx_objectid int null,
  qx_datemodified date null
);


--
-- TASKS
--

create table QX_TASK_DELAYUNIT (
  TASK_DELAYUNIT_ID    INT        NOT NULL,
  TASK_DELAYUNIT_NAME  VARCHAR2(255)    NOT NULL,
constraint PK_QX_TASK_DELAYUNIT primary key (TASK_DELAYUNIT_ID)
);



create table QX_TASK_STATUS (
  TASK_STATUS_ID    INT        NOT NULL,
  TASK_STATUS_NAME  VARCHAR2(255)    NOT NULL,
constraint PK_QX_TASK_STATUS primary key (TASK_STATUS_ID)
);


create table QX_SCRIPT (
  SCRIPT_ID            INT           NOT NULL,
  SCRIPT_NAME          VARCHAR2(255)  NOT NULL,
  SCRIPT_DESCRIPTION   VARCHAR2(1022) NULL,
  SCRIPT_CLASS_NAME    VARCHAR2(255)  NOT NULL,
  CREATOR_ID           INT           NOT NULL,
  DATE_MODIFIED        DATE      NOT NULL,
constraint PK_QX_SCRIPT primary key (SCRIPT_ID)
);

create unique index IDX_QX_SCRIPT_NAME       on QX_SCRIPT (SCRIPT_NAME);
create unique index IDX_QX_SCRIPT_CLASS_NAME on QX_SCRIPT (SCRIPT_CLASS_NAME);


create index IDX_QX_SCRIPT_CREATOR      on QX_SCRIPT (CREATOR_ID);



create table QX_TASK (
  TASK_ID             INT          NOT NULL,
  TASK_NAME           VARCHAR2(255) NOT NULL,
  TASK_STATUS_ID      INT          NOT NULL,
  SCRIPT_ID           INT          NOT NULL,
  CREATOR_ID          INT          NOT NULL,
  DATE_CREATED        DATE     NOT NULL,
  FIRST_START         DATE,
  NEXT_START          DATE,
  REPEAT_SIGN         NUMBER(2)   DEFAULT 0,
  AUTO_START          NUMBER(2)   DEFAULT 0,
  DELAY               INT,
  TASK_DELAYUNIT_ID   INT,
  LOG_PROCESS_ID      NUMBER(38),
constraint PK_QX_TASK primary key (TASK_ID),
constraint FK_QX_TASK_QX_TASK_STATUS  foreign key (TASK_STATUS_ID)    references QX_TASK_STATUS (TASK_STATUS_ID),
constraint FK_QX_TASK_QX_TDU          foreign key (TASK_DELAYUNIT_ID) references QX_TASK_DELAYUNIT (TASK_DELAYUNIT_ID),
constraint FK_QX_TASK_QX_SCRIPT       foreign key (SCRIPT_ID)         references QX_SCRIPT (SCRIPT_ID)
);

create index IDX_QX_TASK_CREATOR      on QX_TASK (CREATOR_ID);
create index IDX_QX_TASK_STATUS       on QX_TASK (TASK_STATUS_ID);
create index IDX_QX_TASK_DELAYUNIT    on QX_TASK (TASK_DELAYUNIT_ID);
create index IDX_QX_TASK_SCRIPT       on QX_TASK (SCRIPT_ID);



--
-- LOCKING
--

create table QX_LOCK (
  PKEY                 INT        NOT NULL,
  TABLE_NAME           VARCHAR2(255)    NOT NULL,
  RECORD_ID            VARCHAR2(100)    NOT NULL,
  RECORD_ID2           VARCHAR2(100)    NOT NULL,
  RECORD_ID3           VARCHAR2(100)    NOT NULL,
  RECORD_ID4           VARCHAR2(100)    NOT NULL,
  SESSION_ID           VARCHAR2(255),
  FOCUS_ID             VARCHAR2(255),
  FOCUS_INSTANCE       INT,
  USER_ID              INT,
  USER_TYPE_ID         INT,
  CREATED              DATE,
constraint PK_QX_LOCK primary key (PKEY),
constraint FK_QX_LOCK_QX_USER_TYPE foreign key (USER_TYPE_ID) references QX_USER_TYPE (USER_TYPE_ID)
);

create unique index IDX_LOCK_TABLE_RECORDS on QX_LOCK (
  TABLE_NAME,
  RECORD_ID,
  RECORD_ID2,
  RECORD_ID3,
  RECORD_ID4
);

create index IDX_QX_LOCK_SESSION   on QX_LOCK (SESSION_ID);
create index IDX_QX_LOCK_FOCUS     on QX_LOCK (FOCUS_ID);
create index IDX_QX_LOCK_USER      on QX_LOCK (USER_ID);
create index IDX_QX_LOCK_USER_TYPE on QX_LOCK (USER_TYPE_ID);



--
-- SEARCH
--

create table QX_SEARCH (
  SEARCH_ID     INT        NOT NULL,
  NAME          VARCHAR2(255)    NOT NULL,
  OWNER_ID      INT,
  TIMESTAMP     DATE    NOT NULL,
  FOCUS_ID      VARCHAR2(255)    NOT NULL,
  TAB_ID        VARCHAR2(255)    NOT NULL,
  FORM_ID       VARCHAR2(255)    NOT NULL,
  BODY          CLOB        NOT NULL,
constraint PK_QX_SEARCH primary key (SEARCH_ID)
);

create index IDX_QX_SEARCH_OWNER on QX_SEARCH (OWNER_ID);
create index IDX_QX_SEARCH_FOCUS on QX_SEARCH (FOCUS_ID);
create index IDX_QX_SEARCH_TAB   on QX_SEARCH (TAB_ID);
create index IDX_QX_SEARCH_FORM  on QX_SEARCH (FORM_ID);


--
-- MISC
--
create table QX_STATES (
    pkey              int             not null,
    name              VARCHAR2(255)    not null,
    state_id          VARCHAR2(2)      not null,
    constraint PK_QX_STATES primary key (pkey)
);

create table QX_MAIL_PROVIDER (
    PKEY int not null,
    PROVIDER_ID VARCHAR2(50),
    PROVIDER_NAME VARCHAR2(50),
constraint PK_QX_MAIL_PROVIDER primary key (PKEY)
);

create table QX_AUTO_REPLY (
    PKEY int not null,
    NAME VARCHAR2(255) not null,
    DESCRIPTION VARCHAR2(255) null,
    SUCCESSFUL_SUBJECT VARCHAR2(255) not null,
    SUCCESSFUL_BODY CLOB null,
    UPDATE_SUBJECT VARCHAR2(255) not null,
    UPDATE_BODY CLOB null,
    FAIL_SUBJECT VARCHAR2(255) not null,
    FAIL_BODY CLOB null,
    CREATED_BY int not null,
    CREATED_DATE date,
constraint PK_QX_AUTO_REPLY primary key (PKEY),
constraint FK_CREATED_BY_QX_USERS foreign key (CREATED_BY) references QX_USERS (PKEY)
);

create table QX_ACCOUNT (
    PKEY int not null,
    OWNER_ID int null,
    WORKGROUP_ID int null,
    ACCOUNT_NAME VARCHAR2(255) not null,
    ACCOUNT_PROVIDER int not null,
    EMAIL_SERVER VARCHAR2(255) not null,
    EMAIL_SERVER_PORT VARCHAR2(50) null,
    EMAIL_SERVER_FOLDER VARCHAR2(50) null,
    EMAIL_PASSWORD VARCHAR2(255) not null,
    EMAIL_BOX_NAME VARCHAR2(255) null,
    LOGIN_NAME VARCHAR2(50) not null,
    LASTRECEIVE_DATE date,
    MAX_MAIL_SIZE int null,
    MAX_INBOX_SIZE int null,
    DELETE_EMAIL_FLAG int null,
    UNREAD_EMAIL_FLAG int null,
    UNREAD_OUTEMAIL_FLAG int null,
    ACTIVE_ACC int null,
    VALID_ACC int null,
    DELAY int null,
    NEXT_CHECK_DATE date,
    NOTES CLOB,
    AUTOREPLY_ID int null,
    CREATED_BY int not null,
    CREATED_DATE date,
    REPLY_EMAIL VARCHAR2(255),
constraint PK_QX_ACCOUNT primary key (PKEY),
constraint FK_ACCNT_QX_WORKGROUP foreign key (WORKGROUP_ID) references QX_WORKGROUP (PKEY),
constraint FK_ACCNT_QX_MAIL_PROVIDER foreign key (ACCOUNT_PROVIDER) references QX_MAIL_PROVIDER (PKEY),
constraint FK_ACCNT_CREATED_BY_QX_USERS foreign key (CREATED_BY) references QX_USERS (PKEY),
constraint FK_ACCNT_ACCOUNT_PROVIDER foreign key (ACCOUNT_PROVIDER) references QX_MAIL_PROVIDER (PKEY),
constraint FK_ACCNT_OWNER_QX_USERS foreign key (OWNER_ID) references QX_USERS (PKEY),
constraint FK_ACCNT_QX_AUTO_REPLY foreign key (AUTOREPLY_ID) references QX_AUTO_REPLY (PKEY)
);

create table QX_EMAIL_ROUTING_RULES (
    PKEY int not null,
    OWNER_ID int null,
    WORKGROUP_ID int null,
    RULE_NAME VARCHAR2(50) not null,
    KEY_WORD VARCHAR2(255) not null,
    FROM_ADDRESS int null,
    TO_ADDRESS int null,
    SUBJECT int null,
    BODY int null,
    AUTOREPLY_ID int null,
    CREATED_BY int not null,
    CREATED_DATE date,
constraint PK_QX_EMAIL_ROUTING_RULES primary key (PKEY),
constraint FK_ROUTING_QX_WORKGROUP foreign key (WORKGROUP_ID) references QX_WORKGROUP (PKEY),
constraint FK_ROUTING_CREATED_BY_QX_USERS foreign key (CREATED_BY) references QX_USERS (PKEY),
constraint FK_ROUTING_OWNER_QX_USERS foreign key (OWNER_ID) references QX_USERS (PKEY)
);

create table QX_EMAIL_ROUTING (
    ACCOUNT_ID int not null,
    ROUTING_RULE int not null,
    constraint PK_QX_EMAIL_ROUTING primary key (ACCOUNT_ID, ROUTING_RULE),
    constraint FK_QX_EMAIL_ROUTING_ACCOUNT foreign key (ACCOUNT_ID) references QX_ACCOUNT(PKEY),
    constraint FK_QX_EMAIL_ROUTING_RULES foreign key (ROUTING_RULE) references QX_EMAIL_ROUTING_RULES(PKEY)
);

create table QX_EMAIL_FILTER (
    PKEY int not null,
    NAME VARCHAR2(50) not null,
    FROM_ADDRESS int null,
    TO_ADDRESS int null,
    SUBJECT int null,
    BODY int null,
    WORD VARCHAR2(255) not null,
    CREATED_BY int not null,
    CREATED_DATE date,
constraint PK_QX_EMAIL_FILTER primary key (PKEY),
constraint FK_EMAIL_FILTER_QX_USERS foreign key (CREATED_BY) references QX_USERS (PKEY)
);

create table QX_FILTER_ACCOUNT (
    ACCOUNT_ID int not null,
    EMAIL_FILTER int not null,
constraint PK_QX_FILTER_ACCOUNT primary key (ACCOUNT_ID, EMAIL_FILTER),
constraint FK_QX_FILTER_ACCNT_ACCOUNT foreign key (ACCOUNT_ID) REFERENCES QX_ACCOUNT(PKEY),
constraint FK_QX_FILTER_ACCNT_FILTER foreign key (EMAIL_FILTER) REFERENCES QX_EMAIL_FILTER(PKEY)
);

create table QX_INBOX (
    PKEY int not null,
    MESSAGE_TYPE int default 0 not null, -- 0 is email message, 1 - alert
    MESSAGE_SENDER int null,
    MESSAGE_SENDER_EMAIL VARCHAR2(1024),
    RECIPIENT_ID int null,
    WORKGROUP_ID int null,
    EMAIL_TO VARCHAR2(1024) null,
    EMAIL_CC VARCHAR2(1024) null,
    SUBJECT VARCHAR2(1024) null,
    MESSAGE CLOB,
    SENT_DATE date,
    SERVER_RECEIVED_DATE date,
    RECEIVED_DATE date,
    OBJECT_TYPE int null,
    OBJECT_ID int null,
    REPLIED_DATE date,
    PROCESS_ID int null,
    ACCOUNT_ID int null,
    DIGEST_SRC VARCHAR2(512),
    DELETE_FLAG int default 0 not null , -- 0 existed, 1 - deleted
constraint PK_QX_INBOX primary key (PKEY),
constraint FK_QX_INBOX_SENDER_QX_USERS foreign key (MESSAGE_SENDER) references QX_USERS(PKEY),
constraint FK_QX_INBOX_RECIPIENT_QX_USERS foreign key (RECIPIENT_ID) references QX_USERS(PKEY),
constraint FK_QX_INBOX_QX_WORKGROUP foreign key (WORKGROUP_ID) references QX_WORKGROUP(PKEY),
constraint FK_QX_INBOX_QX_VIEW_OBJ_TYPES foreign key (OBJECT_TYPE) references QX_VIEW_OBJECT_TYPES(PKEY),
constraint FK_QX_INBOX_QX_ACCOUNT foreign key (ACCOUNT_ID) references QX_ACCOUNT(PKEY)
);

create view QX_DD_OBJECT_TYPE as
    select
        ENUM_VALUE as PKEY,
        ENUM_LABEL as NAME
    from QX_FIELDINFO
    where (TABLE_NAME = 'qx_inbox') and (COLUMN_NAME = 'object_type')
;

create view qx_user_roles as
select distinct r.role_id, u.user_id
  from qx_role r, qx_role_to_users u
  where r.role_id = u.role_id
union all
select r.role_id, g.employeegroup
  from qx_role r, qx_workgroup_to_role wr, qx_groupmember g
  where r.role_id = wr.role_id
    and wr.workgroup_id = g.workgroupgroup
;

create table QX_EMAIL_SERVICE_LOG (
    PKEY int NOT NULL ,
    MESSAGE_ID int NULL ,
    MESSAGE_TYPE int NULL ,
    E_TO VARCHAR2(255) NULL ,
    E_FROM VARCHAR2(255) NULL ,
    E_SUBJECT VARCHAR2(1022) NULL ,
    MESSAGE_LEVEL int NULL ,
    CREATED date NULL ,
    OWNER_ID int NULL ,
    WORKGROUP_ID int NULL ,
    OBJECT_TYPE int null,
    OBJECT_ID int null,
    FILTER_ID int NULL ,
    ROUTING_ID int NULL ,
    DESCRIPTION VARCHAR2(1022) NULL ,
    ACCOUNT_ID int NULL ,
constraint PK_QX_EMAIL_SERVICE_LOG primary key (PKEY)
);
