/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */

package com.queplix.core.integrator.security;

import java.io.Serializable;
import java.util.HashMap;

/**
 * This class represents the user logon session.
 *
 * @author Kozmin Sergey
 * @since 16.01.2007
 */
public class LogonSession implements Serializable {
    // User, role, session ID.
    private final User user;
    private final String sessionID;

    // Access statistics data.
    private String host;
    private String remoteAddress;
    
    private String portalType;

    // Optional parameters.
    private HashMap<String, Object> params = new HashMap<String, Object>();

    /**
     * Creates a new logon session.
     *
     * @param user      User object
     * @param sessionID session ID
     */
    public LogonSession(User user, String sessionID) {
        this.user = user;
        this.sessionID = sessionID;
    }

    /**
     * User VO getter.
     *
     * @return user value object
     */
    public User getUser() {
        return user;
    }

    /**
     * Session ID getter.
     *
     * @return unique session ID
     */
    public String getSessionID() {
        return sessionID;
    }

    /**
     * Host name getter.
     *
     * @return host name
     */
    public String getHost() {
        return host;
    }

    /**
     * Host name setter.
     *
     * @param host host name
     */
    public void setHost(String host) {
        this.host = host;
    }

    /**
     * Remote host address getter.
     *
     * @return remote host address
     */
    public String getRemoteAddress() {
        return remoteAddress;
    }

    /**
     * Remote host address setter.
     *
     * @param remoteAddress remote host address
     */
    public void setRemoteAddress(String remoteAddress) {
        this.remoteAddress = remoteAddress;
    }

    /**
     * Optional parameter getter.
     *
     * @param name optional parameter name
     * @return optional parameter
     */
    public Object getParameter(String name) {
        return params.get(name);
    }

    /**
     * Optional parameter setter.
     *
     * @param name  optional parameter name
     * @param param name optional parameter value
     */
    public void addParameter(String name, Object param) {
        params.put(name, param);
    }

    public String toString() {
        StringBuffer sb = new StringBuffer();
        sb.append("[user = ").append(user);
        sb.append(", sessionID = ").append(sessionID);
        sb.append(", host = ").append(host);
        sb.append(", remoteAddress = ").append(remoteAddress);
        sb.append(", params = ").append(params).append("]");
        return sb.toString();
    }

    public String getPortalType() {
        return portalType;
    }

    public void setPortalType(String portalType) {
        this.portalType = portalType;
    }

}
