/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */

package com.queplix.core.modules.config.utils;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * Stores and manages view objects with ability to sort it by types
 *
 * @author Sergey Kozmin
 * @since 25.01.2007
 */
public class ViewObjectsStructure {
    private Map<String, ViewObject> fullObjects;

    private Map<String, ViewObject> focuses;
    private Map<String, ViewObject> subFocuses;
    private Map<String, ViewObject> tabs;
    private Map<String, ViewObject> forms;


    public ViewObjectsStructure() {
        fullObjects = new HashMap<String, ViewObject>();
        focuses = new HashMap<String, ViewObject>();
        subFocuses = new HashMap<String, ViewObject>();
        tabs = new HashMap<String, ViewObject>();
        forms = new HashMap<String, ViewObject>();
    }

    public ViewObjectsStructure(Map<String, ViewObject> initialObjects) {
        this();
        addAll(initialObjects.values());
    }

    public void addAll(Collection<ViewObject> viewObjects) {
        for(ViewObject viewObject : viewObjects) {
            add(viewObject);
        }
    }

    public void add(ViewObject viewObject) {
        String name = viewObject.getName();
        fullObjects.put(name, viewObject);
        switch(viewObject.getType()) {
            case FOCUS: {
                focuses.put(name, viewObject);
                break;
            }
            case SUB_FOCUS: {
                subFocuses.put(name, viewObject);
                break;
            }
            case TAB: {
                tabs.put(name, viewObject);
                break;
            }
            case FORM: {
                forms.put(name, viewObject);
                break;
            }
        }
    }

    public void remove(ViewObject viewObject) {
        if(viewObject != null) {
            String name = viewObject.getName();
            fullObjects.remove(name);
            switch(viewObject.getType()) {
                case FOCUS: {
                    focuses.remove(name);
                    break;
                }
                case SUB_FOCUS: {
                    subFocuses.remove(name);
                    break;
                }
                case TAB: {
                    tabs.remove(name);
                    break;
                }
                case FORM: {
                    forms.remove(name);
                    break;
                }
            }
        }
    }

    public void remove(String name) {
        remove(fullObjects.remove(name));
    }

    public Map<String, ViewObject> getFullObjects() {
        return Collections.unmodifiableMap(fullObjects);
    }

    public Map<String, ViewObject> getFocuses() {
        return Collections.unmodifiableMap(focuses);
    }

    public Map<String, ViewObject> getSubFocuses() {
        return Collections.unmodifiableMap(subFocuses);
    }

    public Map<String, ViewObject> getTabs() {
        return Collections.unmodifiableMap(tabs);
    }

    public Map<String, ViewObject> getForms() {
        return Collections.unmodifiableMap(forms);
    }
}
