/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */

package com.queplix.core.modules.eqlext.actions;

import com.queplix.core.error.GenericSystemException;
import com.queplix.core.jxb.entity.Dataset;
import com.queplix.core.jxb.entity.Efield;
import com.queplix.core.jxb.entity.Entity;
import com.queplix.core.modules.config.jxb.CustomField;
import com.queplix.core.modules.eql.EQLObject;
import com.queplix.core.modules.eql.error.UserQueryParseException;
import com.queplix.core.modules.eqlext.error.EfieldTransformException;
import com.queplix.core.modules.eqlext.utils.FieldConverter;
import com.queplix.core.utils.log.AbstractLogger;

/**
 * <p>Abstract Action Class</p>
 *
 * @author [ALB] Baranov Andrey
 * @version $Revision: 1.1.1.1 $ $Date: 2005/09/12 15:30:34 $
 */

abstract class AbstractAction
        extends AbstractLogger
        implements Action {

    // action context
    protected ActionContext ctx;

    // ------------------ PUBLIC METHODS -----------------------

    public void setActionContext(ActionContext ctx) {
        this.ctx = ctx;
    }

    // ---------------- PROTECTED ABSTRACT METHODS -------------

    /**
     * Get array of Efield objects take part in query
     *
     * @return array of Efield objects
     */
    protected abstract Efield[] getFields();

    /**
     * Get array of Dataset objects take part in query
     *
     * @return array of Dataset objects
     */
    protected abstract Dataset[] getDatasets();

    // ------------------ PROTECTED METHODS --------------------

    /**
     * Convert user string object to EQL object
     *
     * @param entity Entity object
     * @param field  Efield object
     * @param s      given string
     * @return EQL object
     * @throws UserQueryParseException
     */
    protected final EQLObject convertString2EQLObject(
            Entity entity,
            Efield field,
            String s)
            throws UserQueryParseException {

        try {
            return FieldConverter.string2EQLObject(ctx.getSC(), entity, field,
                    s);
        } catch (EfieldTransformException ex) {
            throwUserQueryParseException(entity, field, s, ex);
            return null;
        }
    }

    /**
     * Convert EQL object to user string object
     *
     * @param entity Entity object
     * @param field  Efield object
     * @param eqlObj given object
     * @return user string
     */
    protected final String convertEQLObject2String(
            Entity entity,
            Efield field,
            EQLObject eqlObj) {

        try {
            return FieldConverter.EQLObject2String(ctx.getSC(), entity, field,
                    eqlObj);
        } catch (EfieldTransformException ex) {
            ERROR(ex);
            throw new GenericSystemException(ex);
        }
    }

    /**
     * Throws UserQueryParseException exception.
     *
     * @param entity Entity object
     * @param field  Efield object
     * @param value  incorrect field value
     * @param ex     caused by
     * @throws UserQueryParseException
     */
    protected final void throwUserQueryParseException(
            Entity entity,
            Efield field,
            String value,
            EfieldTransformException ex)
            throws UserQueryParseException {

        String entityName = entity.getName();
        String fieldName = field.getName();
        CustomField customField = ctx.getCustomField(entityName, fieldName);
        String caption = customField.getCaption();

        throw new UserQueryParseException(value, entityName, caption, null, ex);
    }

    protected final void throwUserQueryParseException(
            Entity entity,
            Efield field,
            String value)
            throws UserQueryParseException {

        String entityName = entity.getName();
        String fieldName = field.getName();
        CustomField customField = ctx.getCustomField(entityName, fieldName);
        String caption = customField.getCaption();

        throw new UserQueryParseException(value, entityName, caption);
    }
}
