/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */

package com.queplix.core.modules.services;

import com.queplix.core.error.ErrorHelper;
import com.queplix.core.error.GenericSystemException;
import com.queplix.core.integrator.security.AccessRightsManager;
import com.queplix.core.integrator.security.LogonSession;
import com.queplix.core.modules.services.utils.log.impl.MailLogPublisher;
import com.queplix.core.utils.log.AbstractLogger;
import com.queplix.core.utils.log.Log;

import java.io.Serializable;

/**
 * Custom action runner class.
 *
 * @author [ALB] Baranov Andrey
 * @author [ONZ] Oleg N. Zhovtanyuk
 * @version $Revision: 1.1.1.1 $ $Date: 2005/09/12 15:30:58 $
 * @see Action
 * @see XAAction
 */
public final class ActionRunner
        implements Serializable {

    // ========================================================== Initialization

    // Logger.
    private static AbstractLogger log = Log.getLog(ActionRunner.class);

    // Private constructor - blocks instantiation.
    private ActionRunner() {
    }

    // ========================================================== Public methods


    /**
     * Runs the action in the given context.
     *
     * @param action  action to run
     * @param context context to use
     * @return result object
     */
    public static Serializable run(Action action, ActionContext context) {

        // Log the action data.
        if(log.getLogger().isInfoEnabled()) {
            log.INFO("Running action: " + action);
        }

        // Run the action.
        Serializable result = null;
        try {
            action.init(context);
            result = action.perform();
        } catch (Exception ex) {
            throwSystemException(ex, action.getClass());
        }

        // Ok.
        return result;

    }

    /**
     * Runs the action in the given XA context.
     *
     * @param action  action to run
     * @param context XA context to use
     * @return result object
     */
    public static Serializable runAsXA(Action action, XAActionContext context) {

        // Log the action data.
        if(log.getLogger().isInfoEnabled()) {
            log.INFO("Running action: " + action);
        }

        // Run the action.
        Serializable result = null;
        try {
            action.init(context);
            result = context.getScriptManager().startAction(action);
        } catch (Exception ex) {
            throwSystemException(ex, action.getClass());
        }

        // Ok.
        return result;

    }

    // ========================================================== Public methods

    // Throws the given exception:
    // 	a) sends the error message by e-mail from admin,
    //  b) throws the exception wrapped in GenericSystemException

    private static void throwSystemException(Throwable t, Class actionClass)
            throws GenericSystemException {

        // Send e-mail.
        try {
            LogonSession ls = AccessRightsManager.getSystemLogonSession();
            new MailLogPublisher(ls, actionClass).ERROR(t);
        } catch (Exception ex) {
            ex.printStackTrace();
        }

        // Throw the exception.
        ErrorHelper.throwSystemException(t, log);

    }

}
