/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */

package com.queplix.core.modules.services.utils.log;

/**
 * The Level class defines a set of standard logging levels.
 * <p/>
 * The levels in ascending order are:
 * <ul>
 * <li>ERROR</li>
 * <li>WARNING</li>
 * <li>INFO</li>
 * <li>DEBUG</li>
 * </ul>
 *
 * @author [ONZ] Oleg N. Zhovtanyuk
 * @author [ALB] Andrey Baranov
 * @version $Revision: 1.1.1.1 $ $Date: 2005/09/12 15:31:02 $
 */

public class Level
        implements java.io.Serializable {

    // ================================================================== Fields

    // The non-localized name of the level.
    private final String name;

    // The integer value of the level.
    private final int value;

    // ========================================================== Initialization

    /**
     * The level to log errors.
     */
    public static final Level ERROR = new Level("ERROR", 0);

    /**
     * The level to warn on non-fatal errors.
     */
    public static final Level WARNING = new Level("WARNING", 1);

    /**
     * The default level - use for informational messages.
     */
    public static final Level INFO = new Level("INFO", 2);

    /**
     * The level for debug messages.
     */
    public static final Level DEBUG = new Level("DEBUG", 3);

    /**
     * Create a named Level with a given integer value.
     *
     * @param name  the name of the Level.
     * @param value an integer value for the level.
     */
    protected Level(String name, int value) {
        this.name = name;
        this.value = value;
    }

    // ========================================================== Public methods

    /**
     * Level string name getter.
     *
     * @return level name
     */
    public String getName() {
        return name;
    }

    /**
     * Level integer value getter.
     *
     * @return level integer value
     */
    public final int intValue() {
        return value;
    }

    /**
     * Parse a level name string into a Level.
     *
     * @param name string to be parsed (non-null)
     * @return parsed value
     */
    public static Level parse(String name) {

        if(name == null) {
            throw new IllegalArgumentException("Level is null");
        }

        if(name.equalsIgnoreCase("ERROR")) {
            return ERROR;
        } else if(name.equalsIgnoreCase("WARNING")) {
            return WARNING;
        } else if(name.equalsIgnoreCase("INFO")) {
            return INFO;
        } else if(name.equalsIgnoreCase("DEBUG")) {
            return DEBUG;
        } else {
            throw new IllegalArgumentException("Bad log level '" + name + "'");
        }
    }

    // ======================================================= Inherited methods

    /* (non-Javadoc)
     * @see java.lang.Object#equals(Object)
     */

    public boolean equals(Object o) {
        if(o == this) {
            return true;
        } else if(!(o instanceof Level)) {
            return false;
        } else {
            Level l = (Level) o;
            return (value == l.value);
        }
    }

    /* (non-Javadoc)
     * @see java.lang.Object#hashCode()
     */
    public int hashCode() {
        return value;
    }

    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    public final String toString() {
        return name;
    }
}
