/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */

package com.queplix.core.modules.services.utils.log.impl;

import com.queplix.core.modules.services.utils.log.AbstractLogHandler;
import com.queplix.core.modules.services.utils.log.AbstractLogRecord;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Timestamp;

/**
 * Base LOG handler
 *
 * @author [ALB] Andrey Baranov
 * @version $Revision: 1.1.1.1 $ $Date: 2005/09/12 15:31:02 $
 */

public class BaseLogHandler
        extends AbstractLogHandler {

    // ----------------------------------------------------- Constants

    /**
     * Main log table name (for primary key extraction).
     */
    public static final String LOG_TABLE = "QX_LOG";

    /**
     * SQL to add a record to the main log.
     */
    public static final String ADD_LOG_SQL =
            "INSERT INTO QX_LOG "
                    + "( LOG_ID, LOG_PROCESS_ID, LOG_TYPE_ID, LOG_CLASS_ID, USER_ID, USER_TYPE_ID, DESCRIPTION, STAMP ) "
                    + "VALUES ( ?, ?, ?, ?, ?, ?, ?, ? )";

    // ----------------------------------------------------- Public methods

    /**
     * Create new log record.
     *
     * @param record the record to insert
     */
    public final void insert(AbstractLogRecord record) {

        // check level
        int level = record.getLevel().intValue();
        if(!canLog(level)) {
            return;
        }

        // do insert
        Connection con = null;

        try {
            con = sqlWrapper.doConnection();
            doInsert(con, record);

        } catch (SQLException ex) {
            ERROR(ex);

        } finally {
            sqlWrapper.closeConnection(con);
        }
    }

    /*
     * No javadoc
     * @see AbstractLogHandler#doInsert
     */
    public long doInsert(Connection con, AbstractLogRecord record)
            throws SQLException {

        BaseLogRecord lr = (BaseLogRecord) record;

        // insert new record
        PreparedStatement stat = null;
        long pkey = sqlWrapper.getNextKey(con, LOG_TABLE);

        try {
            stat = sqlWrapper.doPreparedStatement(con, ADD_LOG_SQL);
            stat.setLong(1, pkey);
            stat.setLong(2, lr.getProcessID());
            stat.setInt(3, lr.getLevel().intValue());
            stat.setInt(4, lr.getClassID());
            stat.setLong(5, lr.getLogonSession().getUser().getUserID());
            stat.setInt(6,
                    lr.getLogonSession().getUser().getAuthenticationType());
            stat.setString(7, lr.getMessage());
            stat.setTimestamp(8, new Timestamp(lr.getTime()));

            sqlWrapper.executeUpdate(stat);

        } finally {
            sqlWrapper.closeConnection(stat);
        }

        return pkey;
    }
}
