/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */

package com.queplix.core.utils;

import com.queplix.core.error.GenericSystemException;
import com.queplix.core.utils.dao.AbstractPropertyFactory;

import java.io.IOException;
import java.io.InputStream;
import java.util.Locale;
import java.util.Properties;
import java.util.TimeZone;

/**
 * Helper class for QX (system) property manipulation.
 *
 * @author [ALB] Baranov Andrey
 * @author [ONZ] Oleg N. Zhovtanyuk
 * @version $Revision: 1.3 $ $Date: 2006/06/02 13:43:17 $
 */
public final class SystemHelper {

    // ======================================== System properties initialization

    // System properties file.
    private static final String FILE_NAME = "qx.properties";

    // Current class.
    private static final Class CLASS = SystemHelper.class;

    // Load system properties.
    private static final Properties props = new Properties();

    static {
        InputStream is = AbstractPropertyFactory.loadSysPropertiesAsStream(
                CLASS, FILE_NAME);
        try {
            props.load(is);
        } catch (IOException ex) {
            ex.printStackTrace();
            throw new GenericSystemException(ex);
        }
    }

    // =============================================================== Constants

    // Production/development mode switch property.
    private static final String MODE = "qx.mode";

    // Production/development modes markers.
    private static final String MODE_DEV = "dev";
    private static final String MODE_PROD = "prod";

    // ======================================================= Main properties

    // System properties.

    public static final String SYSTEM_LANG = getProperty("system.lang", true);
    public static final String SYSTEM_COUNTRY = getProperty("system.country",
            true);
    public static final TimeZone SYSTEM_TIMEZONE = TimeZone.getTimeZone(
            getProperty("system.timezone", true));
    public static final Locale SYSTEM_LOCALE = new Locale(SYSTEM_LANG,
            SYSTEM_COUNTRY);

    // System DB properties.

    public static final String IGNORE_LOWER = getProperty("system.ignoreLower",
            false);

    // Default properties (for a new user).

    public static final String DEFAULT_LANGUAGE = getProperty("default.lang",
            true);
    public static final String DEFAULT_COUNTRY = getProperty("default.country",
            true);
    public static final TimeZone DEFAULT_TIMEZONE = TimeZone.getTimeZone(
            getProperty("default.timezone", true));
    public static final Locale DEFAULT_LOCALE = new Locale(DEFAULT_LANGUAGE,
            DEFAULT_COUNTRY);
    public static final String DEFAULT_DATE_PATTERN = getProperty(
            "default.dpattern", false);
    public static final String DEFAULT_TIME_PATTERN = getProperty(
            "default.tpattern", false);

    // ========================================================== Public methods

    /**
     * Detects development or production mode (default is production)
     *
     * @return true if in production mode
     */
    public static final boolean isProductionMode() {
        return System.getProperty(MODE, MODE_PROD).equals(MODE_PROD);
    }

    /**
     * Gets the system property by name.
     *
     * @param name     property name
     * @param required is property required or not
     * @return property value
     */
    public static final String getProperty(String name, boolean required) {
        String value = (props == null) ? null:(String) props.get(name);
        if(required && StringHelper.isEmpty(value)) {
            throw new GenericSystemException(
                    "Can't get property '" + name + "' from file '" + FILE_NAME
                            + "'.");
        }
        return value;
    }

    /* Prints the RAM info to the standard output. */
    public static final String getRAMinfo() {
        Runtime rt = Runtime.getRuntime();
        return "RAM: " + rt.totalMemory() + " bytes total, " + rt.freeMemory()
                + " bytes free.";
    }
}
