/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */

package com.queplix.core.utils.cache;

import com.queplix.core.utils.log.AbstractLogger;
import com.queplix.core.utils.log.Log;
import org.jboss.cache.CacheException;
import org.jboss.cache.Fqn;
import org.jboss.cache.GlobalTransaction;
import org.jboss.cache.TreeCache;

import java.util.HashMap;
import java.util.Map;

/**
 * <p>JBoss TreeCache wrapper.</p>
 *
 * @author [ALB] Baranov Andrey
 * @version 1.0
 */

public final class JBossCacheWrapper
        extends TreeCache {

    // Logger.
    private AbstractLogger logger = Log.getLog(getClass());

    // Map of listeners.
    private Map map = new HashMap();

    // Name.
    private String name;

    /**
     * Constructor.
     *
     * @param name unique cache name
     * @throws Exception
     */
    public JBossCacheWrapper(String name)
            throws Exception {

        super();
        this.name = name;
    }

    /**
     * Adds new Event listener for events for node <code>fqn</code>.
     *
     * @param listener JBossCacheEventListener
     * @param fqn      The fully qualified name of the node.
     */
    public void addListener(JBossCacheEventListener listener, String fqn) {
        map.put(Fqn.fromString(fqn), listener);

        logger.INFO("Added new listener to listen events for node: " + fqn);
    }

    /**
     * Remove Event listener for events for node <code>fqn</code>.
     *
     * @param fqn The fully qualified name of the node.
     */
    public void removeListener(String fqn) {
        map.remove(Fqn.fromString(fqn));
    }

    /*
     * No javadoc
     * @see TreeCache#_put( GlobalTransaction, Fqn, Map, boolean, boolean )
     */
    public void _put(GlobalTransaction tx, Fqn fqn, Map data,
                     boolean create_undo_ops, boolean erase_contents)
            throws CacheException {

        super._put(tx, fqn, data, create_undo_ops, erase_contents);

        // Call listener.
        JBossCacheEventListener listener = (JBossCacheEventListener) map.get(
                fqn);
        if(listener != null) {
            listener.cacheUpdated(data);
        }
    }

    /*
     * No javadoc
     * @see TreeCache#_put( GlobalTransaction, Fqn, Object, Object, boolean )
     */
    public Object _put(GlobalTransaction tx, Fqn fqn, Object key, Object value,
                       boolean create_undo_ops)
            throws CacheException {

        Object o = super._put(tx, fqn, key, value, create_undo_ops);

        // Call listener.
        JBossCacheEventListener listener = (JBossCacheEventListener) map.get(
                fqn);
        if(listener != null) {
            listener.cacheObjectUpdated(key, value);
        }

        return o;
    }

    /*
     * No javadoc
     * @see TreeCache#_remove( GlobalTransaction, Fqn, Object, Object, boolean )
     */
    public Object _remove(GlobalTransaction tx, Fqn fqn, Object key,
                          boolean create_undo_ops)
            throws CacheException {

        Object o = super._remove(tx, fqn, key, create_undo_ops);

        // Call listener.
        JBossCacheEventListener listener = (JBossCacheEventListener) map.get(
                fqn);
        if(listener != null) {
            listener.cacheObjectRemoved(key);
        }

        return o;
    }

    /*
     * No javadoc
     * @see Object#toString
     */
    public String toString() {
        return "Name: " + name + "; " + super.toString();
    }
}
