/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */
package com.queplix.core.client.common.ui;

import com.google.gwt.user.client.ui.ClickListener;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.SourcesClickEvents;
import com.google.gwt.user.client.ui.Widget;
import com.queplix.core.client.common.event.Event;
import com.queplix.core.client.common.event.EventSource;

import java.util.ArrayList;

/**
 * Base class of all widgets that have multiple clickable events.
 *
 * @author Sultan Tezadov
 * @since 24 Nov 2006
 */
public class ButtonSet extends Composite {
    private EventSource eventSource;
    private ArrayList widgets;
    private ArrayList events;
    private boolean enabled;
    private int clickedIndex = -1;
    private ClickListener innerListener;

    public ButtonSet() {
        enabled = true;
        eventSource = new EventSource(this);
        widgets = new ArrayList();
        events = new ArrayList();
        innerListener = new ClickListener() {
            public void onClick(Widget sender) {
                if(enabled) {
                    clickedIndex = widgets.indexOf(sender);
                    Event event = (Event) events.get(clickedIndex);
                    onEvent(event);
                }
            }
        };
    }

    /**
     * No widget constructor. Use when you need a code only class rather then a widget.
     *
     * @param noWidget if true, this Composite will be initialized with dummy widget
     */
    public ButtonSet(boolean noWidget) {
        this();
        if(noWidget) {
            initWidget(new Label("")); // init with a dummy widget
        }
    }

    public void addButton(Event event, SourcesClickEvents widget) {
        addButton(event, widget, true);
    }
    
    public void addButton(Event event, SourcesClickEvents widget, boolean append) {
        widget.addClickListener(innerListener);
        if(append) {
            widgets.add(widget);
            events.add(event);
        } else {
            widgets.add(0, widget);
            events.add(0, event);
        }
    }

    public void removeButton(int index) {
        ((SourcesClickEvents) widgets.remove(index)).removeClickListener(
                innerListener);
        events.remove(index);
    }

    public void removeButton(SourcesClickEvents widget) {
        int index = widgets.indexOf(widget);
        if(index != -1) {
            widget.removeClickListener(innerListener);
            events.remove(index);
            widgets.remove(widget);
        }
    }

    public void removeAllButtons() {
        for(int i = 0; i < widgets.size(); i++) {
            SourcesClickEvents widget = (SourcesClickEvents) widgets.get(i);
            widget.removeClickListener(innerListener);
        }
        widgets.clear();
        events.clear();
    }

    public boolean isEnabled() {
        return enabled;
    }

    public void setEnabled(boolean enabled) {
        this.enabled = enabled;
    }

    public int getClickedIndex() {
        return clickedIndex;
    }

    public EventSource getEventSource() {
        return eventSource;
    }

    public int getIndex(Event event) {
        return events.indexOf(event);
    }

    public Event getEvent(int index) {
        return (Event) events.get(index);
    }

    public int getButtonsCount() {
        return events.size();
    }

    public SourcesClickEvents getWidget(Event event) {
        int index = getIndex(event);
        if(index > -1) {
            return (SourcesClickEvents) widgets.get(index);
        }
        return null;
    }

    public SourcesClickEvents getWidget(int index) {
        return (SourcesClickEvents) widgets.get(index);
    }

    public void onEvent(Event event) {
        eventSource.fireEventGeneratedByUser(event);
    }
}
