/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */

package com.queplix.core.client.common.ui.grid;

import com.queplix.core.client.app.vo.GridData;
import com.queplix.core.client.app.vo.GridMeta;
import com.queplix.core.client.app.vo.SortField;
import com.queplix.core.client.app.vo.SortFieldImpl;

import java.util.ArrayList;

/**
 * Provides default actions implementation for {@link com.queplix.core.client.common.ui.grid.GridModel}
 * interface, based on {@link com.queplix.core.client.app.vo.GridData} class and
 * {@link com.queplix.core.client.app.vo.GridMeta} class.
 *
 * @author Sergey Kozmin
 * @since 28.03.2007
 */
public abstract class BaseGridModel implements GridModel {
    private GridDataProvider dataProvider;
    private ArrayList gridListeners = new ArrayList();
    private SortFieldImpl sortField = null;

    public BaseGridModel() {
    }

    public int getModelRowCount() {
        if(getData() != null) {
            return getData().getRows().length;
        } else {
            return 0;
        }
    }

    public int getModelColumnCount() {
        if(getMeta() != null) {
            return getMeta().getColumnsCount();
        } else {
            return 0;
        }
    }

    public GridDataProvider getGridData() {
        return dataProvider;
    }

    public void addGridModelListener(GridModelListener listener) {
        gridListeners.add(listener);
    }

    public void removeGridModelListener(GridModelListener listener) {
        gridListeners.remove(listener);
    }

    public SortFieldImpl getSortFieldImpl() {
        if(sortField == null) {
            sortField = new SortFieldImpl();
        }
        return sortField;
    }

    public SortField getSortField() {
        return getSortFieldImpl();
    }

    public SortColumn getSortColumn() {
        SortColumn res = null;
        if(sortField != null) {
            int fieldIndex = getMeta().getFieldIndex(sortField.getFieldID());
            if(fieldIndex > -1) { // found
                int columnIndex = getMeta().getColumns().getSelectedIndex(
                        fieldIndex);
                if(columnIndex > -1) { // found
                    res = new SortColumnImpl(columnIndex,
                            sortField.getSortOrder());
                }
            }
        }
        return res;
    }

    public void fireDataStructureChanged() {
        for(int i = 0; i < gridListeners.size(); i++) {
            GridModelListener listener = (GridModelListener) gridListeners.get(
                    i);
            listener.gridStructureChanged();
        }
    }

    public void fireDataChanged() {
        for(int i = 0; i < gridListeners.size(); i++) {
            GridModelListener listener = (GridModelListener) gridListeners.get(
                    i);
            listener.gridDataChanged();
        }
    }

    public void fireRowDataChanged(int row) {
        for(int i = 0; i < gridListeners.size(); i++) {
            GridModelListener listener = (GridModelListener) gridListeners.get(
                    i);
            listener.gridRowDataChanged(row);
        }
    }

    public void fireRowsWereInserted() {
        for(int i = 0; i < gridListeners.size(); i++) {
            GridModelListener listener = (GridModelListener) gridListeners.get(
                    i);
            listener.gridRowsWereInserted();
        }
    }

    public void setDataProvider(GridDataProvider dataProvider) {
        this.dataProvider = dataProvider;
    }

    protected abstract GridData getData();

    protected abstract GridMeta getMeta();
}
