/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */
package com.queplix.core.client.common.ui.grid;

import com.google.gwt.user.client.DOM;
import com.google.gwt.user.client.ui.ComplexPanel;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.Grid;
import com.google.gwt.user.client.ui.HTMLTable;
import com.google.gwt.user.client.ui.Widget;
import com.queplix.core.client.common.StringUtil;

/**
 * FireFox implementation of {@link GridElementsStrategy}
 *
 * @author Sergey Kozmin
 * @since 16 Nov 2006
 */
public class GridElementsStrategyImplFF extends GridElementsStrategyImpl {
    public void setCellValue(Grid grid, int row, int column, Widget cellValue) {
        grid.setWidget(row, column, cellValue);
    }

    /**
     * Create div wrapper, to be able set width to it. By default FF doesn't allow to set width less
     * than last unwrapped word has.
     */
    public void setCellValue(Grid grid, int row, int column, String cellValue) {
        grid.setWidget(row, column, new BoxWrapper(cellValue));
    }

    public void setHeaderCellValue(Grid grid, int row, int column,
                                   String cellValue, Boolean sortBy) {
        grid.setWidget(row, column, new HeaderWrapper(column, cellValue,
                sortBy));
    }

    /**
     * Perform all rows resizing, in ff need do that.
     */
    public void setOneRowColumnWidth(Grid grid, int row, int column,
                                     int sizeInPixels, int delta) {
        setAllRowsColumnWidth(grid, column, sizeInPixels, delta);
    }

    /**
     * Resize header columns. Header contains {@link HeaderWrapper}
     */
    public void setHeaderColumnWidth(Grid grid, int column, int sizeInPixels,
                                     int delta) {
        grid.setWidth(StringUtil.pixelToSize(grid.getOffsetWidth() + delta));
        grid.getCellFormatter().setWidth(0, column, StringUtil.pixelToSize(
                sizeInPixels));
    }

    /**
     * Here we resize only first row, because on the initialization step
     * we set all other cell's width to 5px (can do less if needed)
     */
    public void setAllRowsColumnWidth(Grid grid, int column, int sizeInPixels,
                                      int delta) {
        String size = StringUtil.pixelToSize(sizeInPixels);

        for(int i = 0; i < grid.getRowCount(); i++) {
            ((BoxWrapper) grid.getWidget(i, column)).setElementWidth(
                    sizeInPixels);
        }

        grid.setWidth(StringUtil.pixelToSize(grid.getOffsetWidth() + delta));
        grid.getCellFormatter().setWidth(0, column, size);
    }

    /**
     * Set all cells width to 5px, because of performance issues.
     * Doing so we are able to resize only first row.
     */
    public void tableInitialized(Grid grid) {
        HTMLTable.CellFormatter cellFormatter = grid.getCellFormatter();
        for(int j = 0; j < grid.getColumnCount(); j++) {
            for(int i = 1; i < grid.getRowCount(); i++) {
                cellFormatter.setWidth(i, j, "5px");
            }
        }
    }

    private static final class BoxWrapper extends ComplexPanel {
        public BoxWrapper() {
            setElement(DOM.createDiv());
            DOM.setStyleAttribute(getElement(), "textAlign", "left");
            DOM.setStyleAttribute(getElement(), "overflow", "hidden");
            DOM.setStyleAttribute(getElement(), "white-space", "normal");
        }

        public BoxWrapper(String text) {
            this();
            setText(text);
        }

        public BoxWrapper(Widget internalWidget) {
            this();
            setWidget(internalWidget);
        }

        private void setWidget(Widget w) {
            add(w);
        }

        private void setText(String text) {
            if(StringUtil.isStringEmpty(text)) {
                DOM.setInnerHTML(getElement(), "&nbsp;");
            } else {
                DOM.setInnerText(getElement(), text);
            }
        }

        public void setElementWidth(int width) {
            DOM.setStyleAttribute(getElement(), "width", width + "px");
        }
    }

    private final class HeaderWrapper extends Composite {
        public HeaderWrapper(int columnIndex, String text, Boolean sortBy) {
            initWidget(GridElementsStrategyImplFF.this.getHeaderCell(
                    columnIndex, new BoxWrapper(text), sortBy));
        }
    }
}
