/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */

package com.queplix.core.client.controls.checkbox;

import com.google.gwt.user.client.ui.ClickListener;
import com.google.gwt.user.client.ui.Widget;
import com.queplix.core.client.common.ui.ButtonData;
import com.queplix.core.client.common.ui.IconButton;

/**
 * Class represents internal view realisation for the Checkbox widget.
 *
 * @author Vasily Mikhailitchenko
 * @since 15.09.2006
 */
class QCheckBoxViewImpl extends QCheckBoxView implements ClickListener {

    // =========== User actions notification for controller implementation
    public static interface UserActionsListener {
        public void onUserActions();
    }

    private UserActionsListener userActionsListener; // the only listener will be the controller implementation

    void addUserActionsListener(UserActionsListener listener) {
        userActionsListener = listener;
    }

    void fireUserActions() {
        if(userActionsListener != null) {
            userActionsListener.onUserActions();
        }
    }
    // ========== End of user actions event

    private static final ButtonData NULL_STATE =
            new ButtonData(null, "Unspecified", "checkbox/checkbox_null.gif");
    private static final ButtonData TRUE_STATE =
            new ButtonData(null, "Selected", "checkbox/checkbox_true.gif");
    private static final ButtonData FALSE_STATE =
            new ButtonData(null, "Not selected", "checkbox/checkbox_false.gif");
    private static final String CHECKBOX_IMAGE_STYLE = "image_button";

    private IconButton checky;
    private QCheckBoxModelImpl model;

    private boolean isEnabled;

    QCheckBoxViewImpl(QCheckBoxModelImpl model, int layout) {
        super(model, layout);
        this.model = model;
        initializeGUI(model.getMeta().getCaption());
    }

    public void onModelMetaChanged() {
        captionChanged();
    }

    public void onModelDataChanged() {
        super.onModelDataChanged();
        stateChanged();
    }

    private void initializeGUI(String fieldCaption) {
        checky = new IconButton(NULL_STATE);
        checky.setStyleName(CHECKBOX_IMAGE_STYLE);
        setCaption(fieldCaption);

        this.updateButtonState();

        checky.addClickListener(this);
        captionLabel.addClickListener(this);

        addToLabelPanel(checky);
        addToLabelPanel(emptyButton);
        initPanel();
    }

    private void updateButtonState() {
        if(null == model.getData().isChecked()) {
            checky.setButtonState(NULL_STATE);
        } else if(model.getData().isChecked().booleanValue()) {
            checky.setButtonState(TRUE_STATE);
        } else {
            checky.setButtonState(FALSE_STATE);
        }
    }

    public void stateChanged() {
        updateButtonState();
    }

    public void captionChanged() {
        if(!getCaption().equalsIgnoreCase(model.getMeta().getCaption())) {
            setCaption(model.getMeta().getCaption());
        }
    }

    protected void setEnabled(boolean isEnabled) {
        this.isEnabled = isEnabled;
    }

    public void onClick(Widget sender) {
        if(isEnabled) {
            fireUserActions();
        }
    }

    protected int getButtonWidth() {
        return checky.getOffsetWidth() + emptyButton.getOffsetWidth() - 1;
    }

}
