/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */
package com.queplix.core.client.controls.datefield;

import com.queplix.core.client.controls.QFormElementControllerImpl;
import com.queplix.core.client.controls.QFormElementModel;
import com.queplix.core.client.controls.QFormElementView;

import java.util.Date;

/**
 * Class represents QDateField controller.
 *
 * @author Sergey Kozmin, Aliaksandr Melnik
 * @since 14 September 2006
 */
class QDateFieldControllerImpl extends QFormElementControllerImpl implements
        QDateFieldController,
        QDateFieldViewImpl.UserActionsListener,
        DateParser.DataParserListener,
        DateFormatter.DataFormatterListener {
    private QDateFieldViewImpl view;
    private QDateFieldModelImpl model;

    private DateParser dateParser = new DateParser();
    private DateFormatter dateFormater = new DateFormatter();

    public QDateFieldControllerImpl(QDateFieldViewImpl view,
                                    QDateFieldModelImpl model) {
        this.view = view;
        this.model = model;
        view.addUserActionsListener(this);
        dateParser.addDataParserListener(this);
        dateFormater.addDataFormatterListener(this);
    }

    public void onUserActions(String stringToParse) {
        Events.DATE_PARSE_EVENT.setData(new Boolean(true));
        getEventSource().fireEvent(Events.DATE_PARSE_EVENT);
        dateParser.parse(stringToParse);
    }

    public void parsed(Date date) {
        model.getData().setStartDate(date);
        if(view.isSearchMode()) {
            model.getData().setEndDate((Date) date.clone());
        } else {
            model.getData().setEndDate(null);
        }
        view.parseSucceed();
        Events.DATE_PARSE_EVENT.setData(new Boolean(false));
        getEventSource().fireEvent(Events.DATE_PARSE_EVENT);
    }

    public void parsed(Date startDate, Date endDate) {
        model.getData().setStartDate(startDate);
        model.getData().setEndDate(endDate);
        view.parseSucceed();
        Events.DATE_PARSE_EVENT.setData(new Boolean(false));
        getEventSource().fireEvent(Events.DATE_PARSE_EVENT);
    }

    public void parseFailed() {
        view.parseFailed();
        Events.DATE_PARSE_EVENT.setData(new Boolean(false));
        getEventSource().fireEvent(Events.DATE_PARSE_EVENT);
    }

    public void forcedDateFormat() {
        onUserActions(model.getData().getStartDate(),
                model.getData().getEndDate());
    }

    public void onUserActions(Date startDate, Date endDate) {
        if(endDate == null) {
            dateFormater.format(startDate);
        } else {
            dateFormater.format(startDate, endDate);
        }
    }

    public void formated(String formatedDate) {
        model.getData().setFormatedDate(formatedDate);
        view.formated();
    }

    public void formatFailed() {
        //method does nothing, previous string stays
    }

    public QFormElementModel getModel() {
        return this.model;
    }

    public QFormElementView getView() {
        return this.view;
    }
}
