/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */

package com.queplix.core.client.controls.multiselect;

import com.google.gwt.user.client.ui.Button;
import com.google.gwt.user.client.ui.ClickListener;
import com.google.gwt.user.client.ui.Widget;
import com.queplix.core.client.app.vo.BaseFieldMeta;
import com.queplix.core.client.app.vo.SubsetData;
import com.queplix.core.client.app.vo.SubsetItemMeta;
import com.queplix.core.client.app.vo.SubsetMeta;
import com.queplix.core.client.app.vo.uisettings.DialogUISettings;
import com.queplix.core.client.common.event.EventListener;
import com.queplix.core.client.common.ui.OkayCancelPopup;
import com.queplix.core.client.common.ui.subsetselect.CheckboxSubsetSelect;
import com.queplix.core.client.controls.QFormElement;

/**
 * Class represents internal view realisation for the date field widget.
 *
 * @author Rustem Nizamiev
 * @since 14.09.2006, 18:54:37
 */
class QMultiSelectViewImpl extends QMultiSelectView implements ClickListener {
    private static final String DEFAULT_WIDTH = "100%";
    private static final String DEFAULT_HEIGHT = "220px";

    private QMultiSelectModelImpl model;
    private Button showButton;

    private OkayCancelPopup dialog;
    private CheckboxSubsetSelect subset;

    private SubsetItemMeta[] displayedItems;

    public QMultiSelectViewImpl(QMultiSelectModelImpl modelIn) {
        super(modelIn, QFormElement.HORIZONTAL);
        this.model = modelIn;
        initializeGUI(model.getMeta().getCaption());
    }

    public void onModelMetaChanged() {
        SubsetItemMeta[] newItems = model.getMeta().getAvailableChoises()
                .getItems();
        if(!BaseFieldMeta.isItemsEquals(displayedItems, newItems)) {
            setMeta();
        }

        captionChanged();
        setupTitle();
    }

    public void onModelDataChanged() {
        super.onModelDataChanged();
        updateButtonState();
    }

    public void onDemandModelChanged() {
        showDialog();
    }

    public void showDialog() {
        if(isReportDesignMode()) {
            return;
        }

        initDialog();
        setMeta();
        subset.setData((SubsetData) model.getData().getItemsSelected().clone());
        setupTitle();
        dialog.show(showButton);
    }

    private void setMeta() {
        SubsetMeta newItems = model.getMeta().getAvailableChoises();
        if(subset != null) {
            subset.setMeta(newItems);
        }
        displayedItems = newItems.getItems();
    }

    private void setupTitle() {
        if(dialog != null) {
            String postfix = " : ";
            if(isSearchMode()) {
                postfix += "Search";
            } else if(isEditMode() || isNewMode()) {
                postfix += "Edit";
            } else if(isDisabled()) {
                postfix += "View";
            }
            dialog.setCaption(getModel().getBaseMeta().getCaption() + postfix);
        }
    }

    private void initDialog() {
        if(dialog == null) {
            dialog = new OkayCancelPopup("");
            subset = new CheckboxSubsetSelect();
            subset.setSize(DEFAULT_WIDTH, DEFAULT_HEIGHT);
            if(getViewMode() == MODE_DISABLED) {
                subset.setEnable(false);
            }
            dialog.setWidget(subset);
            dialog.setUISettings(getModel().getBaseMeta().getUISettings());
            dialog.getEventSource().addEventListener(
                    getEventSource());//redirect events from dialog to our event source
        }
    }

    private void initializeGUI(String fieldCaption) {
        setCaption(fieldCaption);

        showButton = new Button();
        showButton.addClickListener(this);
        showButton.setStyleName(QMultiSelect.BUTTON_CSS_STYLE);
        updateButtonState();

        addToPanel(showButton);

        initPanel();
    }

    private void updateButtonState() {
        String showbuttonCaption;
        if(model.getData().getItemsSelected().getSelectedIDs().length <= 0) {
            showbuttonCaption = "<img src='" + QMultiSelect
                    .BUTTON_EMPTY_IMAGE_PATH + "'>";
        } else {
            showbuttonCaption = "<img src='" + QMultiSelect
                    .BUTTON_FULL_IMAGE_PATH + "'>";
        }
        showButton.setHTML(showbuttonCaption);
    }

    public void onClick(Widget sender) {
        if(sender == showButton) {
            getEventSource().fireEvent(Events.ON_DIALOG_DEMAND);
        }
    }

    public void captionChanged() {
        if(!getCaption().equalsIgnoreCase(model.getMeta().getCaption())) {
            setCaption(model.getMeta().getCaption());
        }
    }

    public void addListener(EventListener listener) {
        getEventSource().addEventListener(listener);
    }

    public SubsetData getDialogItemsSelected() {
        return subset.getData();
    }

    protected void setEnabled(boolean isEnabled) {
        if(subset != null) {
            subset.setEnable(isEnabled);
        }
    }

    protected void setClientWidth(String clientWidth) {
        //resising isn't supported for multiselect
    }

    public int getClientWidth() {
        return 0;
    }

    protected DialogUISettings getUISettings() {
        return (dialog != null) ? dialog.getUISettings():null;
    }
}
