/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */

package com.queplix.core.utils.www;

import com.queplix.core.utils.log.AbstractLogger;
import com.queplix.core.utils.log.Log;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;

/**
 * Base class for all HTTP servlets.
 *
 * @author [ALB] Baranov Andrey
 * @author [ONZ] Oleg N. Zhovtanyuk
 * @version $Revision: 1.1.1.1 $ $Date: 2005/09/12 15:31:24 $
 */

public abstract class AbstractServlet
        extends HttpServlet {

    // ================================================================== Fields

    /**
     * Logger.
     */
    protected AbstractLogger logger;

    // ===================================================== Servlet API methods

    /* (non-Javadoc)
     * @see GenericServlet#init()
     */

    public void init()
            throws ServletException {
        super.init();
        logger = Log.getLog(getClass());
    }

    /* (non-Javadoc)
     * @see GenericServlet#init(ServletConfig)
     */
    public void init(ServletConfig config)
            throws ServletException {
        super.init(config);
        logger = Log.getLog(getClass());
    }

    // ================================================= Other protected methods

    /**
     * Gets EJB remote interface reference.
     *
     * @param jndiName  JNDI name
     * @param homeClass home class
     * @return remote object
     * @throws ServletException
     */
    protected Object getRemoteObject(String jndiName, Class homeClass)
            throws ServletException {
        try {
            return ServletHelper.getRemoteObject(getServletContext(), jndiName,
                    homeClass);
        } catch (Exception ex) {
            throw new ServletException(
                    "Can't get remote object '" + jndiName + "'", ex);
        }
    }

    /**
     * Gets EJB local interface reference.
     *
     * @param jndiName  JNDI name
     * @param homeClass home class
     * @return local object
     * @throws ServletException
     */
    protected Object getLocalObject(String jndiName, Class homeClass)
            throws ServletException {
        try {
            return ServletHelper.getLocalObject(getServletContext(), jndiName,
                    homeClass);
        } catch (Exception ex) {
            throw new ServletException(
                    "Can't get local object '" + jndiName + "'", ex);
        }
    }

    /**
     * Gets EJB home interface reference.
     *
     * @param jndiName  JNDI name
     * @param homeClass home class
     * @return remote home object
     * @throws ServletException
     */
    protected Object getRemoteHome(String jndiName, Class homeClass)
            throws ServletException {
        try {
            return ServletHelper.getRemoteHome(getServletContext(), jndiName,
                    homeClass);
        } catch (Exception ex) {
            throw new ServletException(
                    "Can't get remote home '" + jndiName + "'", ex);
        }
    }

    /**
     * Gets EJB local home interface reference.
     *
     * @param jndiName  JNDI name
     * @param homeClass home class
     * @return local home object
     * @throws ServletException
     */
    protected Object getLocalHome(String jndiName, Class homeClass)
            throws ServletException {
        try {
            return ServletHelper.getLocalHome(getServletContext(), jndiName,
                    homeClass);
        } catch (Exception ex) {
            throw new ServletException(
                    "Can't get local home '" + jndiName + "'", ex);
        }
    }

    /**
     * Determines the action to perform by the HTTP request path.
     *
     * @param request HTTP request
     * @return action name as string
     * @throws ServletException on empty action name
     */
    protected String getAction(HttpServletRequest request)
            throws ServletException {
        String action = request.getPathInfo();
        if(action == null) {
            throw new ServletException(
                    "Can't determine action - no path info in request.");
        }
        return action.substring(1);
    }

}
