/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */

package com.queplix.core.client.app.vo;

import com.queplix.core.client.common.StringUtil;

import java.util.Collection;
import java.util.Iterator;

/**
 * Data container for in-form grid field.
 *
 * @author Sergey Kozmin
 * @since 26.09.2006, 16:18:53
 */
public class InFormGridFieldData extends BaseFieldData {
    /**
     * Data for grid. This object contains string data for grid only.
     * If for example we have several grid columns we can differ them by their ids.
     * But there are second case, when we don't have records in grid, but just a
     * set of fields, and to use it on the server as
     * {@link com.queplix.core.client.app.vo.FieldData} we need to keep it both
     * in the object. This object is {@link #filters}
     */
    private GridData gridData;
    /**
     * Filtering object representation. Collection<FieldData>
     */
    private Collection filters;
    private long recordId;

    public InFormGridFieldData(String fieldID, GridData data) {
        super(fieldID);
        setGridData(data);
    }

    public InFormGridFieldData(String fieldID) {
        this(fieldID, null);
    }

    public InFormGridFieldData() {
        this("");
    }

    public void setRecordId(long pkey){
        recordId = pkey;
    }

    public long getRecordId() {
        return recordId;
    }

    public GridData getGridData() {
        return gridData;
    }

    public void setGridData(GridData gridData) {
        if(gridData != null) {
            this.gridData = gridData;
        } else {
            this.gridData = new GridData(new RowData[0], "empty");
        }
    }

    /**
     * @return filtering objects, if it's null, then there is no filters.
     */
    public Collection getFilters() {
        return filters;
    }

    public void setFilters(Collection filters) {
        this.filters = filters;
    }

    public void clear() {
        gridData.clearRecords();
        if(filters != null) {
            filters.clear();
        }
    }

    public boolean isEmpty() {
        if(filters != null) {
            for(Iterator iterator = filters.iterator(); iterator.hasNext();) {
                FieldData fieldData = (FieldData) iterator.next();
                if(!fieldData.isEmpty()) {
                    return false;
                }
            }
        }
        return gridData.isEmpty();
    }

    public FieldData cloneData() {
        return new InFormGridFieldData(getFieldID(), gridData.cloneGrid());
    }

    public boolean dataEqualsTo(FieldData data) {
        boolean ret = false;
        if(data != null && data.getFieldID().equalsIgnoreCase(getFieldID())) {
            InFormGridFieldData castData = (InFormGridFieldData) data;
            //assume orders of entities are equals, otherwise datas are not
            // equals, the order does mean.
            ret = StringUtil.isStringsEqualsIgnoreNulls(castData.getFieldID(),
                    getFieldID()) && castData.gridData.equalsToGrid(gridData);
        }
        return ret;
    }
}
