/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */

package com.queplix.core.modules.alert.utils;

import com.queplix.core.integrator.security.User;
import com.queplix.core.modules.alert.Alert;
import com.queplix.core.modules.alert.AlertBlock;
import com.queplix.core.modules.alert.AlertData;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * Alert VO class.
 *
 * @author [ALB] Baranov Andrey
 * @version $Revision: 1.1.1.1 $ $Date: 2005/09/12 15:30:09 $
 */

public class AlertVO
        implements Alert {

    // ================================================================= Fields

    // Required attributes
    private long alertID;
    private long creatorID;
    private int creatorType;
    private long senderID;
    private int senderType;
    private String message;
    private int severity;
    private Date dateposted;

    // Optional attributes
    private Long recipientID;
    private Integer recipientType;
    private Long workgroupID;
    private Integer tier;
    private boolean toAll;
    private AlertData data;

    // Blocks
    private List blockList;

    // ================================================================= Constructors

    public AlertVO(long _alertID,
                   long _creatorID,
                   int _creatorType,
                   long _senderID,
                   int _senderType,
                   String _message,
                   int _severity,
                   Date _dateposted) {

        alertID = _alertID;
        creatorID = _creatorID;
        creatorType = _creatorType;
        senderID = _senderID;
        senderType = _senderType;
        message = _message;
        severity = _severity;
        dateposted = _dateposted;
    }

    // ================================================================= Overrided Methods

    public long getAlertID() {
        return alertID;
    }

    public long getCreatorID() {
        return creatorID;
    }

    public int getCreatorType() {
        return creatorType;
    }

    public long getSenderID() {
        return senderID;
    }

    public int getSenderType() {
        return senderType;
    }

    public Long getRecipientID() {
        return recipientID;
    }

    public Integer getRecipientType() {
        return recipientType;
    }

    public Long getWorkgroupID() {
        return workgroupID;
    }

    public Integer getTier() {
        return tier;
    }

    public boolean isToAll() {
        return toAll;
    }

    public String getMessage() {
        return message;
    }

    public int getSeverity() {
        return severity;
    }

    public AlertData getData() {
        return data;
    }

    public Date getDateposted() {
        return dateposted;
    }

    public List getBlocks() {
        return blockList;
    }

    /*
     * No javadoc
     * @see Object#equals
     */
    public boolean equals(Object obj) {
        if(obj == this) {
            return true;
        }

        return (obj instanceof AlertVO) && (((AlertVO) obj).alertID == alertID);
    }

    /*
     * No javadoc
     * @see Object#hashCode
     */
    public int hashCode() {
        return (int) alertID;
    }

    /*
     * No javadoc
     * @see Object#toString
     */
    public String toString() {
        return "alertID=" + alertID + "; senderID=" + senderID +
                "; senderType=" + senderType + "; recipientID=" + recipientID +
                "; recipientType=" + recipientType + "; workgroupID="
                + workgroupID +
                "; tier=" + tier + "; toAll=" + toAll + "; dateposted="
                + dateposted +
                "; message=" + message + "; blocks=" + blockList;
    }

    // ================================================================= Public Methods

    /**
     * Message setter.
     *
     * @param _message String
     */
    public void setMessage(String _message) {
        message = _message;
    }

    /**
     * Severity setter.
     *
     * @param _severity int
     */
    public void setSeverity(int _severity) {
        severity = _severity;
    }

    /**
     * Set User as a recipient.
     *
     * @param _recipientID   long
     * @param _recipientType int
     */
    public void setUserRecipient(long _recipientID, int _recipientType) {

        if(workgroupID != null) {
            throw new IllegalStateException(
                    "Workgroup recipient already set. Alert #" + alertID);
        }

        recipientID = new Long(_recipientID);
        recipientType = new Integer(_recipientType);
    }

    /**
     * Set worgroup as a recipient.
     *
     * @param _workgroupID long
     * @param _tier        int
     */
    public void setWorkgroupRecipient(long _workgroupID, Integer _tier) {

        if(recipientID != null) {
            throw new IllegalStateException(
                    "User recipient already set. Alert #" + alertID);
        }

        workgroupID = new Long(_workgroupID);
        tier = _tier;
    }

    /**
     * Set 'To All' flag.
     *
     * @param _toAll boolean
     */
    public void setToAllRecipient(boolean _toAll) {

        if(_toAll && recipientID != null) {
            throw new IllegalStateException(
                    "Not supported when user recipient is used. Alert #"
                            + alertID);
        }

        toAll = _toAll;
    }

    /**
     * Alert Ad-on data setter.
     *
     * @param _data AlertData
     */
    public void setData(AlertData _data) {
        data = _data;
    }

    /**
     * Add new Alert Block.
     *
     * @param _block AlertBlock
     */
    public void addBlock(AlertBlock _block) {
        if(blockList == null) {
            blockList = new ArrayList();
        }
        blockList.add(_block);
    }

    /**
     * Check Alert Block existence.
     *
     * @param _block AlertBlock
     * @return boolean
     */
    public boolean contains(AlertBlock _block) {
        if(blockList == null) {
            return false;
        }
        return blockList.contains(_block);
    }

    /**
     * Checks if <code>user</code> is a sender of Alert.
     *
     * @param user User
     * @return boolean
     */
    public boolean isSender(User user) {
        long senderID = getSenderID();
        int senderType = getSenderType();
        if(user.getUserID() == senderID
                && user.getAuthenticationType() == senderType) {
            return true;
        }
        return false;
    }

    /**
     * Checks if <code>user</code> is an owner of Alert.
     *
     * @param user User
     * @return boolean
     */
    public boolean isOwner(User user) {
        Long recipientID = getRecipientID();
        if(recipientID != null) {
            Integer recipientType = getRecipientType();
            if(user.getUserID() == recipientID &&
                    user.getAuthenticationType() == recipientType) {
                return true;
            }
        }
        return false;
    }

    /**
     * Checks if alert was sent to one of workgroups <code>userGroups</code>.
     * @param userGroups List of UserGroup objects
     * @return boolean
     */
    /*public boolean isOwner( List<Integer> userGroups ) {
        Long workgroupID = getWorkgroupID();
        if( workgroupID != null ) {
            Integer tier = getTier();
            int userGroupsSize = ( userGroups == null ) ? 0 : userGroups.size();
            for( int i = 0; i < userGroupsSize; i++ ) {
                WorkGroup userGroup = ( WorkGroup ) userGroups.get( i );
                long userGroupID = userGroup.getGroupID();
                Integer userGroupTier = userGroup.getTier();
                if( userGroupID == workgroupID.longValue() &&
                    ( tier == null || tier.equals( userGroupTier ) ) ) {
                    return true;
                }
            }
        }
        return false;
    }*/

    /**
     * Checks if alert has 'To All' flag.
     *
     * @return boolean
     */
    public boolean isToAllAlert() {
        boolean toAll = isToAll();
        if(toAll) {
            return true;
        }
        return false;
    }
}
