/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */

package com.queplix.core.modules.eql.parser.generic;

import com.queplix.core.jxb.entity.Efield;
import com.queplix.core.jxb.entity.Entity;
import com.queplix.core.modules.config.utils.EntityHelper;
import com.queplix.core.modules.eql.EQLReqField;
import com.queplix.core.modules.eql.error.EQLException;
import com.queplix.core.modules.eql.error.EQLSystemException;
import com.queplix.core.modules.eql.parser.generic.jj.EQLParserGenericImpl;
import com.queplix.core.utils.log.AbstractLogger;
import com.queplix.core.utils.log.Log;

/**
 * <p>EQL parser abstract class</p>
 *
 * @author Baranov Andrey [ALB]
 * @version $Revision: 1.1.1.1 $ $Date: 2005/09/12 15:30:28 $
 */

public abstract class AbstractEQLParser
        implements InterfaceEQLParser {

    protected AbstractLogger log = Log.getLog(AbstractEQLParser.class);

    /**
     * Get EQL mediator
     *
     * @return EQLIntMediator object
     */
    public abstract EQLIntMediator getMediator();

    /**
     * Get new parser
     *
     * @param mediator EQLIntMediator object
     * @param eql      EQL query
     * @return EQL parser
     */
    public static AbstractEQLParser getParser(EQLIntMediator mediator,
                                              String eql) {
        return new EQLParserGenericImpl(mediator, eql);
    }

    /**
     * Get toplevel entity (first entity in EQL query)
     *
     * @param s <code><entity name>.*</code>
     * @return Entity object
     */
    public Entity getTopLevelEntity(String s) {
        int len = s.length();
        String entityName = s.substring(0, len - 2);
        return getMediator().getSession().findEntity(entityName);
    }

    /**
     * Build EQLReqField object by field id attribute
     *
     * @param fieldId field id attribute
     * @return new EQLReqField object
     * @throws EQLException
     */
    public EQLReqField getReqFieldByFieldId(String fieldId)
            throws EQLException {

        int pos = fieldId.lastIndexOf(".");
        try {
            String entityName = fieldId.substring(0, pos);
            String fieldName = fieldId.substring(pos + 1);

            Entity entity = getMediator().getSession().findEntity(entityName);
            Efield field = EntityHelper.getEfield(fieldName, entity);
            return new EQLReqField(entity, field);

        } catch (StringIndexOutOfBoundsException ex) {
            throw new EQLSystemException("Invalid field '" + fieldId + "'");
        }
    }

    /**
     * Print error message
     *
     * @param msg error string
     */
    public void printError(String msg) {
        log.ERROR(msg);
    }
}
