/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */

package com.queplix.core.modules.eqlext.transforms;

import com.queplix.core.error.GenericSystemException;
import com.queplix.core.jxb.entity.Efield;
import com.queplix.core.utils.dao.AbstractPropertyFactory;

import java.io.InputStream;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Properties;

/**
 * <p>Factory class for working with transform classes</p>
 *
 * @author [ALB] Baranov Andrey
 * @version $Revision: 1.2 $ $Date: 2005/10/26 14:03:51 $
 */

public class EfieldTransformFactory
        implements java.io.Serializable {

// ----------------------- VARIABLES -----------------------

    private static final String FILE_NAME = "transform.properties";
    private static Class CLASS = EfieldTransformFactory.class;

    private static final HashMap transformMap = new HashMap();

    static {

        InputStream is = AbstractPropertyFactory.loadSysPropertiesAsStream(
                CLASS, FILE_NAME);
        Properties props = new Properties();

        try {
            // load properties
            props.load(is);

            // instantiate transforms
            Enumeration en = props.keys();
            while(en.hasMoreElements()) {
                String name = (String) en.nextElement();
                String className = props.getProperty(name);

                EfieldTransform transform = (EfieldTransform) Class.forName(
                        className).newInstance();
                transformMap.put(name, transform);
            }

        } catch (Exception ex) {
            ex.printStackTrace();
            throw new GenericSystemException(ex);
        }
    }

// --------------------- PUBLIC METHODS --------------------

    /**
     * Get EfieldTransform object
     *
     * @param efield Efield object
     * @return transform object
     */
    public static EfieldTransform getEfieldTransform(Efield efield) {

        // is field selectable?
        if(!efield.getSelectable().booleanValue()) {
            return null;
        }

        String transformName = efield.getTransform();
        if(transformName == null) {
            // try to get default transformer
            String datatype = efield.getDatatype().toString();
            transformName = datatype;
        }

        return getEfieldTransform(transformName);
    }

// ------------------- PRIVATE METHODS ---------------------

    /**
     * Constructor
     */
    private EfieldTransformFactory() {
    }

    /**
     * Get EfieldTransform object
     *
     * @param name transform name
     * @return new transform object
     */
    private static EfieldTransform getEfieldTransform(String name) {

        try {
            return (EfieldTransform) transformMap.get(name);

        } catch (Exception ex) {
            throw new GenericSystemException("Can't load transform object '" +
                    name + "' from properties '" +
                    FILE_NAME + "'", ex);
        }
    }
}
