/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */

package com.queplix.core.modules.jeo.gen;

import com.queplix.core.integrator.security.LogonSession;
import com.queplix.core.modules.eql.error.EQLException;
import com.queplix.core.modules.jeo.JEObjectHandler;
import com.queplix.core.modules.jeo.ejb.JEOManagerLocal;

import java.text.MessageFormat;
import java.util.List;

/**
 * Chart JEO handler.
 *
 * @author Michael Trofimov
 */
public class ChartObjectHandler
        extends JEObjectHandler {

    private static final String ENTITY = "chart";

    private static final String SQL_SELECT_BY_ID
            = "select {0}.* where {0}.chart_id = {1}";
    private static final String SQL_SELECT_SYSTEM_CHARTS_BY_NAME
            = "select {0}.* where {0}.name = ''{1}'' and {0}.system_chart = 1";
    private static final String SQL_SELECT_PUBLIC_SYSTEM_CHARTS
            = "select {0}.* where {0}.system_chart = 1 and {0}.public_chart = {1} order by {0}.chart_position";

    public final static class ChartVisibility {
        private ChartVisibility() {
        }
        public static int PRIVATE = 0;
        public static int PUBLIC = 1;
    }
    // ======================================================= Inherited methods

    /* (non-Javadoc)
     * @see JEObjectHandler#getEntityName()
     */

    public String getEntityName() {
        return ENTITY;
    }

    /* (non-Javadoc)
     * @see JEObjectHandler#getObjectClass()
     */
    public Class getObjectClass() {
        return ChartObject.class;
    }

    // ============================================================= EQL methods

    /**
     * Gets the chart JEO handler by ID.
     *
     * @param jeoManager JEOManager EJB local interface
     * @param ls         LogonSession object with security data
     * @param id         Chart ID to search by
     * @return chart JEO handler, or <code>null</code> if none found
     * @throws EQLException
     */
    public static JEObjectHandler selectByID(
            JEOManagerLocal jeoManager, LogonSession ls, long id)
            throws EQLException {

        String eql = MessageFormat.format(
                SQL_SELECT_BY_ID, new Object[]{
                ENTITY,
                id
        });
        List hndList = jeoManager.select(ls, ChartObjectHandler.class, eql);
        return hndList != null ? (JEObjectHandler) hndList.get(0):null;
    }

    /**
     * Gets the chart JEO handler by name.
     *
     * @param jeoManager JEOManager EJB local interface
     * @param ls         LogonSession object with security data
     * @param name       Chart name to search by
     * @return chart JEO handler, or <code>null</code> if none found
     * @throws EQLException
     */
    public static JEObjectHandler selectByName(
            JEOManagerLocal jeoManager, LogonSession ls, String name)
                    throws EQLException {

        checkName(name);

        String eql = MessageFormat.format(
                SQL_SELECT_SYSTEM_CHARTS_BY_NAME, new Object[]{ ENTITY, name });
        List hndList = jeoManager.select(ls, ChartObjectHandler.class, eql);
        return hndList != null ? (JEObjectHandler) hndList.get(0) : null;
    }

    /**
     * Gets chart JEO by ID.
     *
     * @throws EQLException
     */
    public static ChartObject findByID(
            JEOManagerLocal jeoManager, LogonSession ls, long id)
            throws EQLException {

        return getJEObject(selectByID(jeoManager, ls, id), ChartObject.class);
    }

    /**
     * Gets chart JEO by name.
     *
     * @throws EQLException
     */
    public static ChartObject findByName(
            JEOManagerLocal jeoManager, LogonSession ls, String name)
            throws EQLException {
        
        checkName(name);
        return getJEObject(selectByName(jeoManager, ls, name), ChartObject.class);
    }

    /**
     * Returns collection of public system charts (used for dashboard)
     * 
     * @throws EQLException
     */
    public static List<ChartObject> findSystemCharts(
            JEOManagerLocal jeoManager, LogonSession ls)
                    throws EQLException {
        
        return getJEObjectList(
                selectSystemCharts(jeoManager, ls),
                        ChartObject.class);
    }

    /**
     * Returns collection of ChartObject instances that represents system charts
     * 
     * @throws EQLException
     */
    public static List<ChartObject> findSystemCharts(
            JEOManagerLocal jeoManager, LogonSession ls, int visibility)
                    throws EQLException {
        
        return getJEObjectList(
                selectSystemCharts(jeoManager, ls, visibility),
                        ChartObject.class);
    }

    /**
     * Returns collection of public system charts (used for dashboard)
     * 
     * @throws EQLException
     */
    public static List selectSystemCharts(
            JEOManagerLocal jeoManager, LogonSession ls)
                    throws EQLException {
        return selectSystemCharts(jeoManager, ls, ChartVisibility.PUBLIC);
    }

    /**
     * Returns collection of ChartObject instances that represents system charts
     * 
     * @throws EQLException
     */
    public static List selectSystemCharts(
            JEOManagerLocal jeoManager, LogonSession ls, int visibility)
                    throws EQLException {

        String eql = MessageFormat.format(
                SQL_SELECT_PUBLIC_SYSTEM_CHARTS, new Object[]{ENTITY, visibility});
        return  jeoManager.select(ls, ChartObjectHandler.class, eql);
    }

    private static void checkName(String name) {
        if (name == null || name.trim().equals(""))
            throw new IllegalArgumentException("Argument 'name' can't be an empty");
    }

}