/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */

package com.queplix.core.modules.services.utils.log;

import com.queplix.core.error.GenericSystemException;
import com.queplix.core.integrator.security.LogonSession;
import com.queplix.core.modules.services.ejb.LogManager;
import com.queplix.core.modules.services.ejb.LogManagerHome;
import com.queplix.core.utils.JNDINames;

import javax.naming.InitialContext;
import javax.rmi.PortableRemoteObject;
import java.io.Serializable;
import java.rmi.RemoteException;

/**
 * Base log publisher class.
 *
 * @author [ALB] Andrey Baranov
 * @author [ONZ] Oleg N. Zhovtanyuk
 * @version $Revision: 1.1.1.1 $ $Date: 2005/09/12 15:31:02 $
 */
public abstract class AbstractLogPublisher
        implements Serializable {

    // ================================================================== Fields

    // Logon session.
    private LogonSession ls;

    // ========================================================== Initialization

    /**
     * Creates a new log publisher with the given action context and log process ID.
     *
     * @param ls logon session
     */
    public AbstractLogPublisher(LogonSession ls) {
        if(ls == null) {
            throw new NullPointerException("Logon sessoin is NULL");
        }
        this.ls = ls;
    }

    // ========================================================== Public methods

    /**
     * Logs an error message.
     *
     * @param message message to log
     */
    public abstract void ERROR(String message);

    /**
     * Logs a warning.
     *
     * @param message message to log
     */
    public abstract void WARN(String message);

    /**
     * Logs an informational message.
     *
     * @param message message to log
     */
    public abstract void INFO(String message);

    /**
     * Logs a debug message.
     *
     * @param message message to log
     */
    public abstract void DEBUG(String message);

    /**
     * User logon session getter.
     *
     * @return user logon session
     */
    public final LogonSession getLogonSession() {
        return ls;
    }

    // ======================================================= Protected methods

    /**
     * Publishes (i.e. logs) a log record.
     *
     * @param lr log record object
     */
    protected final void publish(AbstractLogRecord lr) {

        try {
            Object o = new InitialContext().lookup(JNDINames.LogManagerRemote);
            LogManagerHome home = (LogManagerHome) PortableRemoteObject.narrow(
                    o, LogManagerHome.class);
            LogManager remote = home.create();
            remote.sendMessage(lr);

        } catch (RemoteException ex) {
            ex.printStackTrace();
            throw new GenericSystemException(
                    "Remote exception: " + ex.getMessage(), ex.detail);

        } catch (Throwable t) {
            t.printStackTrace();
            throw new GenericSystemException(
                    "Unknown exception: " + t.getMessage(), t);
        }
    }

}
