/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */

package com.queplix.core.utils.xml;

import com.queplix.core.error.GenericSystemException;

import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Templates;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Iterator;
import java.util.Map;

/**
 * <p>Abstract translet wrapper implementation</p>
 *
 * @author [ALB] Baranov Andrey
 * @version $Revision: 1.1.1.1 $ $Date: 2005/09/12 15:31:25 $
 */

public abstract class AbstractTransletWrapper
        implements TransletWrapper {

    // ----------------------------------------------------- variables

    // Get TransformerFactory.
    protected TransformerFactory factory;

    // ----------------------------------------------------- public methods

    /**
     * Constructor.
     */
    public AbstractTransletWrapper() {
        factory = TransformerFactory.newInstance();
    }

    /*
     * No javadoc
     * @see TransletWrapper#loadTemplate(InputStream)
     */
    public Templates loadTemplate(InputStream xslStream) {
        try {
            return factory.newTemplates(new StreamSource(xslStream));

        } catch (TransformerException ex) {
            throw new GenericSystemException(
                    "Transformer exception: " + ex.getMessage(), ex);

        } catch (Exception ex) {
            throw new GenericSystemException(
                    "Unknown exception: " + ex.getMessage(), ex);
        }
    }

    /*
     * No javadoc
     * @see TransletWrapper#transform(File,File,Map)
     */
    public OutputStream transform(File xslFile, File xmlFile, Map params) {

        try {
            // Get Transformer.
            Transformer t = factory.newTransformer(new StreamSource(xslFile));

            // Add XSLT global parameters
            t.clearParameters();
            if(params != null) {
                Iterator it = params.keySet().iterator();
                while(it.hasNext()) {
                    String key = (String) it.next();
                    Object value = params.get(key);
                    t.setParameter(key, value);
                }
            }

            // Do transform.
            StreamResult result = new StreamResult(new ByteArrayOutputStream());
            t.transform(new StreamSource(xmlFile), result);
            return result.getOutputStream();

        } catch (TransformerException ex) {
            throw new GenericSystemException(
                    "Transformer exception: " + ex.getMessage(), ex);

        } catch (Exception ex) {
            throw new GenericSystemException(
                    "Unknown exception: " + ex.getMessage(), ex);
        }
    }

    /*
     * No javadoc
     * @see TransletWrapper#transform(Source,Result,Templates,Map)
     */
    public void transform(Source source, Result result, Templates templates,
                          Map params) {
        try {
            // Get Transformer.
            Transformer t = templates.newTransformer();

            // Add XSLT global parameters
            t.clearParameters();
            if(params != null) {
                Iterator it = params.keySet().iterator();
                while(it.hasNext()) {
                    String key = (String) it.next();
                    Object value = params.get(key);
                    t.setParameter(key, value);
                }
            }

            // Do transform.
            t.transform(source, result);

        } catch (TransformerException ex) {
            throw new GenericSystemException(
                    "Transformer exception: " + ex.getMessage(), ex);

        } catch (Exception ex) {
            throw new GenericSystemException(
                    "Unknown exception: " + ex.getMessage(), ex);
        }
    }

}
