/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */
package com.queplix.core.client.controls.memo;

import com.google.gwt.user.client.ui.HTML;
import com.google.gwt.user.client.ui.TextArea;
import com.google.gwt.user.client.ui.Widget;
import com.queplix.core.client.app.vo.uisettings.DialogUISettings;
import com.queplix.core.client.common.StringUtil;
import com.queplix.core.client.common.ui.resizable.Resizable;
import com.queplix.core.client.controls.QElementsCoupler;
import com.queplix.core.client.frames.htmledit.HtmlEditFrame;

/**
 * QMemoViewImpl.
 *
 * @author Sultan Tezadov
 * @since 22 Feb 2007
 */
public abstract class QMemoViewImpl extends QMemoView {
    private static final int TEXTAREA_HEIGHT = 5;
    private static final int TEXTAREA_WIDTH = 30;

    protected static final int CONTROL_TEXTAREA = 0;
    protected static final int CONTROL_HTML_EDITOR = 1;
    protected static final int CONTROL_HTML_VIEWER = 2;

    protected class QueplixTextArea extends TextArea implements Resizable {
        public void setOffsetHeight(int height) {
            super.setHeight(StringUtil.pixelToSize(height));
        }

        public void setOffsetWidth(int width) {
            super.setWidth(StringUtil.pixelToSize(width));
        }
    }

    protected QMemoModelImpl model;
    protected QMemoEventHandlersCollection eventHandlers;
    protected QueplixTextArea textArea;
    protected HtmlEditFrame htmlEditorFrame;
    protected HTML html;

    protected static final QMemoEvent saveEvent = new QMemoEvent(
            QMemoEvent.EVENT_SAVE);

//    protected boolean ignoreTextUpdate;

    public static QMemoViewImpl createInstance(QMemoModelImpl model,
                                               int layout) {
        QMemoViewImpl instance;
        if(model.getMeta().isInline()) {
            instance = new QMemoViewImplInline(model, layout);
        } else {
            instance = new QMemoViewImplPopup(model, layout);
        }
        return instance;
    }

    protected QMemoViewImpl(QMemoModelImpl model, int layout) {
        super(model, layout);
        this.model = model;
        eventHandlers = new QMemoEventHandlersCollection();
        initGUI();
    }

    protected abstract DialogUISettings getUISettings();

    public abstract void showPopup();

    public void addEventHandler(QMemoEventHandler handler) {
        eventHandlers.add(handler);
    }

    public String getText() {
        return textArea.getText();
    }

    public void setText(String newText) {
        newText = StringUtil.nullToEmpty(newText);

        textArea.setText(newText);
        htmlEditorFrame.setText(newText);

        if(HtmlEditFrame.isHTML(newText)) {
            html.setHTML(newText);
        } else {
            html.setHTML("<pre>" + newText + "</pre>");
        }
    }

    protected void initGUI() {
        setCaption(model.getMeta().getCaption());

        textArea = new QueplixTextArea();
        textArea.addStyleName("styled_input");
        textArea.setCharacterWidth(TEXTAREA_WIDTH);
        textArea.setVisibleLines(TEXTAREA_HEIGHT);

        html = new HTML();
        html.setWidth("100%");
        html.setHeight("100%");

        htmlEditorFrame = new HtmlEditFrame(model);
        htmlEditorFrame.setText(model.getData().getText());
    }

    public void onModelMetaChanged() {
        setCaption(model.getMeta().getCaption());
        onModelDataChanged();
    }

    public void onModelDataChanged() {
        super.onModelDataChanged();
        setText(model.getData().getText());
    }

    protected int getControlType() {
        boolean readOnly = model.getBaseMeta().isReadOnly();
        boolean changeMode = isEditMode() || isNewMode();
        if(!readOnly && changeMode) {
            return CONTROL_HTML_EDITOR;
        } else {
            if(isSearchMode()) {
                return CONTROL_TEXTAREA;
            } else {
                return CONTROL_HTML_VIEWER;
            }
        }
    }

    protected void refreshGUIData() {
        switch(getControlType()) {
            case CONTROL_TEXTAREA:
                break;
            case CONTROL_HTML_EDITOR:
                htmlEditorFrame.setText(model.getData().getText());
                htmlEditorFrame.load();
                break;
            case CONTROL_HTML_VIEWER:
                break;
        }
    }

    protected void contentChanged() {
    }

    public Widget getActiveControl() {
        return null;
    }

    public boolean canBeCoupled(int coupleType) {
        return getCaption().length() == 0
                && coupleType == QElementsCoupler.RIGHT_ITEM;
    }

    public void coupleElement(int coupleType) {
        info.removeFromParent();
    }

    public void applyModification() {
    }

    public void revertModification() {
    }
}
