/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */
package com.queplix.core.client.frames.mainframe;

import com.queplix.core.client.app.vo.FamgMeta;
import com.queplix.core.client.app.vo.FieldData;
import com.queplix.core.client.app.vo.FieldOnDemandData;

import java.util.Collection;

/**
 * Contains all operations related directly to form
 *
 * @author Sergey Kozmin
 * @since 16.04.2007
 */
public interface FormOperations {
    
    /**
     * Returns field data object.
     *
     * @param index     form index where element resides
     * @param elementId element id
     * @return field data object. If form wasn't initialized, returns null object.
     * @throws IllegalArgumentException if there is no such field on that form.
     */
    public FieldData getFieldData(FamgMeta.Index index, String elementId)
            throws IllegalArgumentException;

    /**
     * Set data to field. If there is no such field on the form this call
     * will be ignored.
     *
     * @param index form, where field resides.
     * @param data  field data
     */
    public void setFieldData(FamgMeta.Index index, FieldData data);

    /**
     * Set on-demand data for the given form.
     *
     * @param formIndex form index
     * @param data      data
     */
    public void setOnDemandData(FamgMeta.Index formIndex,
                                FieldOnDemandData data);

    /**
     * Set data for form
     *
     * @param fields           fields of the created prototype
     * @param rowId            rowId of the created prototype
     * @param index            form index, that request the prototype
     * @param clearOtherFields should be fileds, not presented be cleared
     */
    public void setFormData(FieldData[] fields, Long rowId,
                            FamgMeta.Index index,
                            boolean clearOtherFields);

    /**
     * Clears the form.
     *
     * @param index               form index
     * @param clearExternalFields should external fields set in descriptor be
     *                            be cleaned (if attribute equals true, then it can cause RPC call to
     *                            server)
     */
    public void clearForm(FamgMeta.Index index, boolean clearExternalFields);

    /**
     * Clear the given field in the given form.
     *
     * @param index   from index where element resides.
     * @param fieldId field id.
     */
    public void clearFormField(FamgMeta.Index index, String fieldId);

    /**
     * Turn form to the needed state. If form is busy (wait for some response)
     * it'll return false.
     *
     * @param index     form index
     * @param formState form state. constant to be get
     *                  from {@link com.queplix.core.client.frames.mainframe.FormState}
     * @return was the form turned to state
     */
    public boolean turnFormToState(FamgMeta.Index index, int formState);

    /**
     * This method finalize all waitings processes, reset internal form
     * state to initial and set form to the given state. Usually you need to
     * call it if you don't wan't (or don't need) to wait for an answer.
     *
     * @param index     form index
     * @param formState state to set.
     */
    public void resetAndSetFormState(FamgMeta.Index index, int formState);

    /**
     * @param index form index
     * @return form state. value is defined in
     *         {@link com.queplix.core.client.frames.mainframe.FormState}
     */
    public int getFormState(FamgMeta.Index index);

    /**
     * Return active record id, selected in form, or null if form is not
     * activated or filled.
     *
     * @param index form index
     * @return active record id
     */
    public Long getSelectedRecordId(FamgMeta.Index index);

    /**
     * Activates the given form.
     *
     * @param index form index
     */
    public void activateForm(FamgMeta.Index index);

    /**
     * Retuns direct form filters, which doesn't depends on form state.
     *
     * @param formIndex index of the requesing form
     * @return not empty form filters list. Can return a null object.
     *         Collection<FieldData>
     */
    public Collection getDirectFormFilters(FamgMeta.Index formIndex);

    /**
     * Set the given button to enabled or disabled state
     *
     * @param formIndex form where button resides
     * @param buttonId  button id
     * @param isEnabled enabled or disable
     */
    public void setButtonEnabled(FamgMeta.Index formIndex, String buttonId,
                                 boolean isEnabled);
}
