/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */
package com.queplix.core.client.frames.mainframe.impl;

import com.queplix.core.client.app.vo.MetaData;
import com.queplix.core.client.frames.mainframe.IMainFrame;
import com.queplix.core.client.frames.mainframe.MainFrameBuilder;
import com.queplix.core.client.frames.mainframe.OperationContext;
import com.queplix.core.client.frames.mainframe.StrategiesFactory;
import com.queplix.core.client.frames.mainframe.ViewPropertiesProvider;

import java.util.Collections;
import java.util.Set;

/**
 * This is the only class that has public access in this package.
 * It creates default implementation of the mainframe, operation context
 * and strategies factory. Also it provides the method for creating it's own
 * strategies factory and mainframe.
 *
 * @author Sergey Kozmin
 * @since 11.04.2007
 */
public class DefaultMainFrameBuilder implements MainFrameBuilder {
    private IMainFrame impl;
    private OperationContext octx;
    private StrategiesFactory strategiesFactory;
    private ViewPropertiesProvider viewPropertiesProvider;

    private MainFrame mf;

    public void initBuilder(MetaData meta) {
        mf = new MainFrame();

        impl = createMainFrame();
        octx = mf;//todo create factoryMethod too

        strategiesFactory = createStrategiesFactory(octx);
        if(strategiesFactory == null) {
            throw new IllegalStateException("Can't have null factory object. ");
        }

        viewPropertiesProvider = createViewPropertiesProvider();

        impl.initMainFrame(meta, strategiesFactory, viewPropertiesProvider);
    }

    public StrategiesFactory getStrategiesFactory() {
        if(strategiesFactory == null) {
            throw new IllegalStateException("Couldn't return factory. "
                    + " Call initBuilder method first. ");
        }
        return strategiesFactory;
    }

    public OperationContext getOperationContext() {
        if(octx == null) {
            throw new IllegalStateException(
                    "Couldn't return operation context. "
                            + " Call initBuilder method first. ");
        }
        return octx;
    }

    public IMainFrame getMainFrame() {
        if(impl == null) {
            throw new IllegalStateException("Couldn't return main frame. "
                    + " Call initBuilder method first. ");
        }
        return impl;
    }

    public ViewPropertiesProvider getViewPropertiesProvider() {
        if(viewPropertiesProvider == null) {
            throw new IllegalStateException(
                    "Couldn't return properties provider. "
                            + " Call initBuilder method first. ");
        }
        return viewPropertiesProvider;
    }

    /**
     * Cannot return null object.
     * This method creates factory. To be overloaded in derived classes.
     *
     * @param ctx operation context
     * @return created factory.
     */
    protected StrategiesFactory createStrategiesFactory(OperationContext ctx) {
        return new DefaultStrategiesFactory(ctx);
    }

    /**
     * Cannot return null object.
     * This method creates main frame. To be overloaded in derived classes.
     *
     * @return mainframe object
     */
    protected IMainFrame createMainFrame() {
        return mf;
    }

    protected ViewPropertiesProvider createViewPropertiesProvider() {
        return new ViewPropertiesProvider() {
            public Set getFormControlListeningEvents(String formId,
                                                     String elementId) {
                return Collections.EMPTY_SET;
            }
        };
    }
}
