/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */
package com.queplix.core.integrator;

import com.queplix.core.client.app.vo.EntityData;
import com.queplix.core.client.app.vo.EntityElement;

import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Set;

/**
 * This class describes inconsistencies, that was found during report approving.
 *
 * @author Sergey Kozmin
 * @since 23.03.2007
 */
class ReportInconsistency {
    static enum InconsistencyType {
        OK(0), WARNING(1), ERROR(2);

        private int errorLevel;

        InconsistencyType(int errorLevel) {
            this.errorLevel = errorLevel;
        }

        public int getErrorLevel() {
            return errorLevel;
        }
    }

    private InconsistencyType type = InconsistencyType.OK;
    /**
     * Form names for which user doesn't have permissions.
     */
    private Set<String> anavailableForms = new HashSet<String>();
    /**
     * Fields that doesn't presented in entity, hence could not be included to report.
     */
    private Collection<EntityElement> anavailableReportFields
            = new LinkedList<EntityElement>();
    /**
     * Fields that doesn't presented in entity, hence filters for which could not be applied.
     */
    private Collection<EntityData> anavailableFilterField
            = new LinkedList<EntityData>();

    public void reset() {
        type = InconsistencyType.OK;

        anavailableForms.clear();
        anavailableReportFields.clear();
        anavailableFilterField.clear();
    }

    public void formNotAvailable(String formTitle) {
        anavailableForms.add(formTitle);
        type = InconsistencyType.ERROR;
    }

    public void reportElementNotAvailable(EntityElement element) {
        anavailableReportFields.add(element);
        if(type.getErrorLevel() < InconsistencyType.WARNING.getErrorLevel()) {
            type = InconsistencyType.WARNING;
        }
    }

    public void reportFilterNotAvailable(EntityData filter) {
        anavailableFilterField.add(filter);
        if(type.getErrorLevel() < InconsistencyType.WARNING.getErrorLevel()) {
            type = InconsistencyType.WARNING;
        }
    }

    public InconsistencyType getType() {
        return type;
    }

    public Set<String> getAnavailableForms() {
        return anavailableForms;
    }

    public void setAnavailableForms(Set<String> anavailableForms) {
        this.anavailableForms = anavailableForms;
    }

    public Collection<EntityElement> getAnavailableReportFields() {
        return anavailableReportFields;
    }

    public void setAnavailableReportFields(
            Collection<EntityElement> anavailableReportFields) {
        this.anavailableReportFields = anavailableReportFields;
    }

    public Collection<EntityData> getAnavailableFilterField() {
        return anavailableFilterField;
    }

    public void setAnavailableFilterField(
            Collection<EntityData> anavailableFilterField) {
        this.anavailableFilterField = anavailableFilterField;
    }
}
