/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */
package com.queplix.qwoss.inbox.utils.filters;

import com.queplix.core.error.GenericSystemException;
import com.queplix.core.modules.inbox.InboxMessage;
import com.queplix.core.modules.inbox.utils.AbstractMailFilter;
import com.queplix.core.modules.inbox.utils.log.AbstractInboxLogPublisher;
import com.queplix.core.modules.inbox.utils.log.RoutingLogPublisher;
import com.queplix.core.modules.jeo.ejb.JEOManagerLocal;
import com.queplix.core.modules.jeo.ejb.JEOManagerLocalHome;
import com.queplix.core.utils.JNDINames;
import com.queplix.core.utils.cache.CacheObjectManager;
import com.queplix.qwoss.inbox.ejb.InboxManagerLocal;
import com.queplix.qwoss.inbox.ejb.InboxManagerLocalHome;
import com.queplix.qwoss.utils.ApplicationHelper;

/**
 * The filter creates Interation object.
 *
 * @author Konstantin Mironov
 * @since 8 Dec 2006
 */
public class InteractionFilter extends AbstractMailFilter {

    // ------------------------------------------------------- Main method

    /*
     * No javadoc
     * @see MailFilter#filterMail
     */

    public boolean filterMail(InboxMessage inboxMessage) {

        inboxPublisher.DEBUG("Email Interaction filter is started...");

        // Init log data.
        String logMessage;
        String logID = makeLogID(inboxMessage.getMessageId());
        Long interactionId;

        InboxManagerLocal local = getLocalInboxManager();

        try {
            interactionId = local.createInteraction(getLogonSession(),
                    inboxMessage);
        } catch (Exception ex) {
            // Logging and re-throwing the exception.
            logMessage = "Interaction filter failed due to the error: " + ex
                    .getMessage();
            inboxPublisher.ERROR(logMessage, inboxMessage);
            ERROR(logID + logMessage);

            throw new GenericSystemException(ex);
        } // try

        // check the interaction
        if(interactionId != null) {
            // if the ticket has already existed
            try {
                if(inboxMessage.getObjectId() != null) {
                    // create link between the new interaction and the existing ticket
                    local.createLinkInteractionTicket(getLogonSession(),
                            interactionId, inboxMessage.getObjectId());
                } else {
                    // set object information
                    inboxMessage.setObjectInfo(interactionId,
                            ApplicationHelper.INTERACTION_OBJECT_TYPE);
                }
            } catch (Exception ex) {
                // Logging and re-throwing the exception.
                logMessage = "Interaction filter failed due to the error: " + ex
                        .getMessage();
                inboxPublisher.ERROR(logMessage, inboxMessage);
                ERROR(logID + logMessage);

                throw new GenericSystemException(ex);
            } // try
        } else {
            // Logging - interaction wasn't created.
            logMessage
                    = "Interaction filter couldn't create an interaction due to the error: Interaction ID is null.";
            inboxPublisher.ERROR(logMessage, inboxMessage);
            ERROR(logID + logMessage);

            throw new GenericSystemException(logMessage);
        } // if (interactionId != null)

        // Ok.
        inboxPublisher.DEBUG("Email Interaction filter was finished...");
        return true;

    } // filterMail()

    // ------------------------------------------------------- Protected methods

    /*
     * No javadoc
     * @see AbstractMailFilter#createInboxLogPublisher
     */

    protected AbstractInboxLogPublisher createInboxLogPublisher() {
        return new RoutingLogPublisher(getLogonSession());
    }

    /**
     * JEOManagerLocal getter.
     *
     * @return JEOManagerLocal
     */
    protected JEOManagerLocal getJEOManager() {
        CacheObjectManager com = new CacheObjectManager();
        return (JEOManagerLocal) com.getLocalObject(JNDINames.JEOManager,
                JEOManagerLocalHome.class);
    } // getJEOManager() : JEOManagerLocal

    /**
     * InboxManagerLocal getter.
     *
     * @return InboxManagerLocal
     */
    protected InboxManagerLocal getLocalInboxManager() {
        CacheObjectManager com = new CacheObjectManager();
        return (InboxManagerLocal) com.getLocalObject(JNDINames.InboxManager,
                InboxManagerLocalHome.class);
    } // getInboxManager() : InboxManagerLocal
} // end of class
