#--
# Copyright (c) 2006 Kinetic Theory, Inc.
#
# Permission is hereby granted, free of charge, to any person obtaining
# a copy of this software and associated documentation files (the
# "Software"), to deal in the Software without restriction, including
# without limitation the rights to use, copy, modify, merge, publish,
# distribute, sublicense, and/or sell copies of the Software, and to
# permit persons to whom the Software is furnished to do so, subject to
# the following conditions:
#
# The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
# LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
# OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
# WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
#++

require 'pathname'

# Generates the component and migration necessary to create the RealApps CMS
class RealappsCmsGenerator < Rails::Generator::Base
  def manifest
    record do |m|

			# Create components/realapps/cms directory
			m.directory "components/realapps/cms"

			# copy controller, authorization and configuration system
			m.file "cms_controller.rb", "components/realapps/cms_controller.rb"
			m.file "cms_authorization.rb", "components/realapps/cms_authorization.rb"
			m.file "cms_authorization_config.rb", "components/realapps/cms_authorization_config.rb"

			# copy model
			m.file "cms_content.rb", "components/realapps/cms_content.rb"

			# copy partials to cms\cms
			Dir.glob("#{source_root}/views/*").each { |f|
				f = File.basename(f)
				m.file "views/#{f}", "components/realapps/cms/#{f}"
			}

			# create directory for stylesheet, etc
			m.directory "public/realapps/cms"

			# copy images and stylesheet to public
			Dir.glob("#{source_root}/html/*").each { |f|
				f = File.basename(f)
				m.file "html/#{f}", "public/realapps/cms/#{f}"
			}

			# Create tests directory
			m.directory "test/unit"
			m.directory "test/functional"
			m.directory "test/fixtures"

			# Loop over directories and files in test/ directory
			Dir.glob("#{source_root}/test/**/*").each { |f|
				if File.file?(f)
					filename = File.basename(f)
					# Gets parent directory of file
					dir = Pathname.new(f).parent.basename
					m.file "test/#{dir}/#{filename}", "test/#{dir}/#{filename}"
				end
			}

			m.conditional_migration 'cms_migration.rb', 'db/migrate', :migration_file_name => "cms_migration"
    end
  end

	# In Rails < 1.1.1, file_name is defind on NamedBase only, and
	# migration_template will fail unless we define it ourselves.
	unless defined?(file_name)
		def file_name; "cms_migration"; end
	end

	class Rails::Generator::Commands::Create
		# Specialized migration template command that will check if the template
		# exists already. If it does, do not migrate.
		def conditional_migration(source, destination, options = {})
			# need to set where our migration directory is, otherwise won't be able to
			# tell if the migration exists already or not.
			migration_directory destination
			# nothing to do if migration already created here.
			if migration_exists?(options[:migration_template_filename] || file_name)
				return
			else
				migration_template(source, destination, options)
			end
		end
	end

	# Need to implement an empty conditional_migration method for the other
	# command available to Generator.
	%w(Rails::Generator::Commands::Destroy Rails::Generator::Commands::List).each { |klass|
		class_eval <<-EOS
			class #{klass}
				def conditional_migration(source, destination, options = {})
					migration_template(source, destination, options)
				end
			end
		EOS
	}
end
