# Model class for accessing CMS content
class Realapps::CmsContent < ActiveRecord::Base

	# ensure create date is always set before saving new content.
	before_create do |content|
		# Notice that ActiveRecord passes the record to the block here - 'self'
		# does not work as expected here.
		content.created = DateTime.now
	end

	# Make sure placement ID is set
	validates_presence_of :placement, :message => " ID must be set."
	# Make sure placement ID given is unique.
	validates_uniqueness_of :placement, :scope => :placement, :message => " ID has already been used."

	# Returns a list of categories and the amount of content
	# associated with each. If no categories are defined, an empty array is returned.
	def self.category_list
		find_by_sql("select category, count(*) as countr from cms_contents where category is not null and category <> '' group by category") || []
	end

	# Gets a list of content in the given category. If category is blank or nil,
	# will return all uncategorized content. If no content exists in the given category,
	# an empty array is returned.
	def self.find_by_category(category)
		if category.blank?
			find(:all, :conditions => "category is null OR category = ''") || []
		else
			find(:all, :conditions => ["category = ?", category]) || []
		end
	end

	# Redefine content_columns to also exclude 'body', since we don't
	# want to display that in our list.
	def self.content_columns
		unless @content_columns
			# "super" returns an array of columns. We can subtract our
			# 'body' column from this array and stash it in a instance variable
			# for caching purposes.
			filtered_content_columns = super
			filtered_content_columns.delete(columns_hash["body"])
			@content_columns = filtered_content_columns
		end

		@content_columns
	end

	# Gets the content from the preview table as specified by the given placement id. If none is found,
	# "" is returned.
	def self.find_preview_content(placement_id)
		begin
			if content = find_by_placement(placement_id)
				content.body
			else
				""
			end
		rescue
			logger.error $!
			"" # ensure empty string is returned regardless of errors.
		end
	end

	# Gets the content from the live table as specified by the given placement id. If none is found,
	# "" is returned.
	def self.find_live_content(placement_id)
		begin
			# Change underlying table name temporarily
			set_table_name "cms_contents_live"
			if content = find_by_placement(placement_id)
				content.body
			else
				""
			end
		rescue
			logger.error $!
			"" # ensure empty string is returned regardless of errors.
		ensure
			# Make sure we are guarnteed to change table name back.
			set_table_name "cms_contents"
		end
	end

	# Copy all content from the cms_contents table to the cms_contents_live
	# table.
	def self.publish_all
		self.connection.transaction do
			self.connection.execute "update cms_contents set published = now()"
			self.connection.execute "delete from cms_contents_live"
			self.connection.execute "insert into cms_contents_live (select * from cms_contents)"
		end
	end
end