require File.dirname(__FILE__) + '/../test_helper'
require 'realapps/cms_controller'

class CmsControllerTest < Test::Unit::TestCase

  AUTHORIZATION_HEADER = 'HTTP_AUTHORIZATION'
  fixtures :cms_contents
  # This line is necessary because the model is in its own namespace, and Rails
  # cannot automatically find it based on the fixture name.
  set_fixture_class :cms_contents => Realapps::CmsContent

  def setup
    @controller = Realapps::CmsController.new
    @request = ActionController::TestRequest.new
    @response = ActionController::TestResponse.new
    # Set up username and password for testing
    Realapps::Cms::Authorization.username = Realapps::Cms::Authorization.password = "foo"
  end

  def teardown
    # clear credentials, if they exist
    @request.env.delete(AUTHORIZATION_HEADER)
  end

  # Test that index cannot be accessed without credentials
  def test_unauthorized_access
    get :index
    assert_response(401, "Unauthorized access allowed: #{@response.response_code}")
  end

  # Test that content CAN be accessed with credentials.
  def test_authorized_access
    add_credentials
    get :index
    assert_response(200, "Authorized access not allowed: #{@response.response_code}")
  end

  def test_access_not_configured
    # Ensures no access is given if username/password not set
    Realapps::Cms::Authorization.username = Realapps::Cms::Authorization.password = nil

    get :index
    assert_response(200, "Did not receive OK response.")
    assert_template("_not_configured", "'Not Configured' message was not displayed.")
  end

  # Test that content can be created.
  def test_create
    add_credentials
    post :create, { :content => { :body => "hello", :category => "hello_category", :placement => "test_add_new", :location_notes => "" } }

    # Make sure we are redirected correctly
    assert_redirected_to({ :action => "list" }, "Failed to create content - not redirected to list afterwards. Error: #{@controller.instance_variable_get("@content").errors.full_messages.inspect}")
    # Make sure content actually exists in model
    assert_not_nil(Realapps::CmsContent.find_by_placement("test_add_new"), "Content was not found in database as expected.")
  end

  # Test that content can be updated.
  def test_update
    add_credentials

    # Configuration determines which method is used to get individual fixtures.
    if self.class.use_instantiated_fixtures
      pre_update_content = @cms_contents["pre_update_content"]
    else
      pre_update_content = cms_contents(:pre_update_content)
    end

    body, category, location_notes = "hello, again", "test_update", "new note"
    post :update, { :content => {:body => body, :category => category, :placement => pre_update_content.placement, :location_notes => location_notes}, :id => pre_update_content.id  }
    assert_redirected_to({ :action => "list" }, "Not redirected to list as expected.")

    post_update_content = Realapps::CmsContent.find_by_placement(pre_update_content.placement)
    assert_not_nil(post_update_content, "Content post-update not found.")
    assert(post_update_content.body == body &&
           post_update_content.category = category &&
           post_update_content.location_notes == location_notes, "Content did update as expected: #{post_update_content.inspect}")
  end

  private

  # Convenience method to add authorization credentials to session
  def add_credentials
    encoded_credentials = Base64.encode64("#{Realapps::Cms::Authorization.username}:#{Realapps::Cms::Authorization.password}")
    @request.env[AUTHORIZATION_HEADER] = "Basic #{encoded_credentials}"
  end
end
