require File.dirname(__FILE__) + '/../test_helper'
require 'realapps/cms_content'

class CmsTest < Test::Unit::TestCase
  fixtures :cms_contents
  # This line is necessary because the model is in its own namespace, and Rails
  # cannot automatically find it based on the fixture name.
  set_fixture_class :cms_contents => Realapps::CmsContent

	def test_nil_placement_id
		content = new_content(:placement => nil)
		content.save
		assert(! content.errors.empty?, "Able to save content with nil placement ID.")
	ensure
		content.destroy
	end

	def test_blank_placement_id
		content = new_content(:placement => "")
		content.save
		assert(! content.errors.empty?, "Able to save content with blank placement ID.")
	ensure
		content.destroy
	end

	def test_whitespace_placement_id
		content = new_content(:placement => "    	\n\t")
		content.save
		assert(! content.errors.empty?, "Able to save content with blank placement ID.")
	ensure
		content.destroy
	end

	def test_duplicate_placement_id
		content1 = new_content(:placement => "foo")
		content1.save
		assert(content1.errors.empty?, "Unexpected error when saving: #{content1.errors.full_messages.join(",")}")

		content2 = new_content(:placement => "foo")
		content2.save
		assert(! content2.errors.empty?, "Able to save content with duplicate placement ID")
	ensure
		content1.destroy
		content2.destroy
	end

	def test_preview_content
		# Verify that content saved is not showing up live, but shows in preview
		content = new_content(:placement => "preview_content", :body => "hello")
		content.save

		assert(! Realapps::CmsContent.find_preview_content("preview_content").blank?, "Did not find preview content as expected.");

		assert(Realapps::CmsContent.find_live_content("preview_content").blank?, "Found live content unexpectedly.");
	ensure
		content.destroy
	end

	def test_publish
		content = new_content(:placement => "preview_to_live_content", :body => "hello")
		content.save

		assert(Realapps::CmsContent.find_live_content("preview_to_live_content").blank?, "Found live content before expected.")

		Realapps::CmsContent.publish_all
		assert(! Realapps::CmsContent.find_live_content("preview_to_live_content").blank?, "Did not find live content after publish.")
	ensure
		content.destroy
		Realapps::CmsContent.publish_all # clears live table
	end

	def test_publish_date
		content = new_content(:placement => "test_publish_date", :body => "hello")
		content.save
		# Reload to ensure we get latest attribute values.
		content = Realapps::CmsContent.find_by_placement("test_publish_date")
		assert(content.published.blank?, "Publish date unexpectedly set.")
		Realapps::CmsContent.publish_all
		# Reload to ensure we get latest attribute values.
		content = Realapps::CmsContent.find_by_placement("test_publish_date")
		assert(! content.published.blank?, "Publish date not set.")
	ensure
		content.destroy
		Realapps::CmsContent.publish_all # clears live table
	end

	def test_created_set
		content = new_content(:placement => "test_create_date_set")
		assert(content.created.blank?, "Create date set before save.")

		content.save
		assert(! content.created.blank?, "Create date not set after save.")
	ensure
		content.destroy
	end

	def test_category_list

    # Need to get list of categories from fixtures file. Instance variable
    # access (@cms_contents) is the only way to iterate over all fixtures. Therefore, we need to
    # load all the fixtures if they have not been loaded yet.
    Fixtures.instantiate_all_loaded_fixtures(self) if ! self.class.use_instantiated_fixtures
		expected_categories = @cms_contents.collect { |k, v| v["category"] }

		categories = Realapps::CmsContent.category_list
		# Ensure every category returned appears in expected_categories list
		category_test = categories.all? { |c|
			expected_categories.include?(c.category)
		}

		assert(category_test, "An expected category was not found.")
	end

	def test_nonexistent_category
		list = Realapps::CmsContent.find_by_category("nonexistent_category")
		assert(list.empty?, "Found content in nonexistent category")
	end

	def test_find_by_category
		list = Realapps::CmsContent.find_by_category("category_one")
		assert(! list.empty?, "Did not find content in category_one as expected")
	end

	def test_content_columns
		# Determine if any content_columns includes body or id
		col_test = Realapps::CmsContent.content_columns.any? { |col|
				["body", "id", ].include?(col)
		}

		assert(! col_test , "Content columns included unexpected columns.")
	end

	def test_blank_categories
		# Ensure content with no category does not appear in categories_list count
		category_list = Realapps::CmsContent.category_list
		content = new_content(:placement => "test_blank_categories")
		content.save

		new_category_list = Realapps::CmsContent.category_list
		category_list.zip(new_category_list) { |lists|
			category_test = (lists[0].countr == lists[1].countr && lists[0].category == lists[1].category)
			assert(category_test, "Categories #{lists[0].category} and #{lists[1].category} found to not match.")
		}
	ensure
		content.destroy
	end

	def test_add_to_category
		category_one_before = Realapps::CmsContent.find_by_category("category_one")
		content = new_content(:placement => "test_add_to_category", :category => "category_one")
		content.save
		category_one_after = Realapps::CmsContent.find_by_category("category_one")
		assert(category_one_before.length == category_one_after.length - 1, "Content did not get added to category_one as expected.")
	ensure
		content.destroy
	end

	private

  # Convenience method which creates content for us. Pass in
  # a hash containing attribute values for the new content.
	def new_content(options = {})
		Realapps::CmsContent.new(options)
	end

end
