/*
Copyright (C) 2001, 2006 United States Government
as represented by the Administrator of the
National Aeronautics and Space Administration.
All Rights Reserved.
*/
package gov.nasa.worldwind.formats.gpx;

import gov.nasa.worldwind.*;

/**
 * @author tag
 * @version $Id: GpxTrackPoint.java 515 2007-01-18 05:09:19Z ericdalgliesh $
 */
public class GpxTrackPoint extends gov.nasa.worldwind.formats.gpx.ElementParser implements gov.nasa.worldwind.TrackPoint
{
    private double latitude;
    private double longitude;
    private double elevation;
    private String time;

    /**
     * @param uri
     * @param lname
     * @param qname
     * @param attributes
     * @throws IllegalArgumentException if attribues is null
     */
    @SuppressWarnings({"UnusedDeclaration"})
    public GpxTrackPoint(String uri, String lname, String qname, org.xml.sax.Attributes attributes)
    {
        super("trkpt");

        //don't validate uri, lname or qname - they aren't used.

        if (attributes == null)
        {
            String msg = WorldWind.retrieveErrMsg("nullValue.org.xml.sax.AttributesIsNull");
            WorldWind.logger().log(java.util.logging.Level.FINE, msg);
            throw new IllegalArgumentException(msg);
        }

        for (int i = 0; i < attributes.getLength(); i++)
        {
            String attrName = attributes.getLocalName(i);
            String attrValue = attributes.getValue(i);
            if (attrName.equalsIgnoreCase("lat"))
            {
                this.latitude = Double.parseDouble(attrValue);
            }
            else if (attrName.equalsIgnoreCase("lon"))
            {
                this.longitude = Double.parseDouble(attrValue);
            }
        }
    }

    @Override
    public void doStartElement(String uri, String lname, String qname, org.xml.sax.Attributes attributes)
        throws org.xml.sax.SAXException
    {
        //don't perform validation here - no parameters are actually used
    }

    /**
     * @param uri
     * @param lname
     * @param qname
     * @throws IllegalArgumentException if <code>lname</code> is null
     * @throws org.xml.sax.SAXException
     */
    @Override
    public void doEndElement(String uri, String lname, String qname) throws org.xml.sax.SAXException
    {
        if (lname == null)
        {
            String msg = WorldWind.retrieveErrMsg("nullValue.LNameIsNull");
            WorldWind.logger().log(java.util.logging.Level.FINE, msg);
            throw new IllegalArgumentException(msg);
        }
        // don't validate uri or qname - they aren't used.

        if (lname.equalsIgnoreCase("ele"))
        {
            this.elevation = Double.parseDouble(this.currentCharacters);
        }
        else if (lname.equalsIgnoreCase("time"))
        {
            this.time = this.currentCharacters.trim();
        }
    }

    public double getLatitude()
    {
        return latitude;
    }

    /**
     * @param latitude
     * @throws IllegalArgumentException if <code>latitude</code> is less than -90 or greater than 90
     */
    public void setLatitude(double latitude)
    {
        if (latitude > 90 || latitude < -90)
        {
            String msg = WorldWind.retrieveErrMsg("generic.angleOutOfRange") + WorldWind.retrieveErrMsg(
                "punctuation.space") + latitude;
            WorldWind.logger().log(java.util.logging.Level.FINE, msg);
            throw new IllegalArgumentException(msg);
        }

        this.latitude = latitude;
    }

    public double getLongitude()
    {
        return longitude;
    }

    /**
     * @param longitude
     * @throws IllegalArgumentException if <code>longitude</code> is less than -180 or greater than 180
     */
    public void setLongitude(double longitude)
    {
        if (longitude > 180 || longitude < -180)
        {
            String msg = WorldWind.retrieveErrMsg("generic.angleOutOfRange") + WorldWind.retrieveErrMsg(
                "punctuation.space") + longitude;
            WorldWind.logger().log(java.util.logging.Level.FINE, msg);
            throw new IllegalArgumentException(msg);
        }

        this.longitude = longitude;
    }

    public double getElevation()
    {
        return elevation;
    }

    public void setElevation(double elevation)
    {
        this.elevation = elevation;
    }

    public String getTime()
    {
        return time;
    }

    /**
     * @param time
     * @throws IllegalArgumentException if <code>time</code> is null
     */
    public void setTime(String time)
    {
        if (time == null)
        {
            String msg = WorldWind.retrieveErrMsg("nullValue.TimeIsNull");
            WorldWind.logger().log(java.util.logging.Level.FINE, msg);
            throw new IllegalArgumentException(msg);
        }
        this.time = time;
    }

    @Override
    public String toString()
    {
        return String.format("(%10.6f\u00B0, %11.6f\u00B0, %10.4g m, %s)", this.latitude, this.longitude,
            this.elevation, this.time);
    }
}
