package gov.nasa.worldwind.formats.rpf;

import gov.nasa.worldwind.formats.nitfs.*;
import java.nio.*;
/*
Copyright (C) 2001, 2007 United States Government
as represented by the Administrator of the
National Aeronautics and Space Administration.
All Rights Reserved.
*/

/**
 * @author lado
 * @version $Id: RpfColorMap Apr 2, 2007 9:43:32 PM
 */
public class RpfColorMap
{
    public int getTableID()
    {
        return tableID;
    }

    public int getHistogramRecordLength()
    {
        return histogramRecordLength;
    }

    public int getHistogramTableOffset()
    {
        return (int)(0xFFFFFFFFL & histogramTableOffset);
    }

    public int getNumOfColorRecords()
    {
        return (int)(0xFFFFFFFFL & numOfColorRecords);
    }

    public int getColorElementLength()
    {
        return (int)(0xFFFFFFFFL & colorElementLength);
    }

    public byte getColor(int colorRec, int bytePosition)
    {
        long idx = colorRec * this.getNumOfColorRecords() * getColorElementLength() + bytePosition ;
        return this.colorMap[(int)idx];
    }



    public byte[] getColorMap()
    {
        return this.colorMap;
    }

    private byte[]  colorMap;
    // private byte[]  histogramMap;

    private int     tableID;
    private long    numOfColorRecords;

    private short   colorElementLength;
    private int     histogramRecordLength;
    private long    colorTableOffset;
    private long    histogramTableOffset;

    public RpfColorMap(java.nio.ByteBuffer buffer, int colormapSubsectionOffset)
    {
        this.parseRpfColorOffsetRecord(buffer);
        // now let's load color map and histogram
        int saveOffset = buffer.position();
        this.loadColorMaps(buffer, colormapSubsectionOffset);
        // ok, we can skip histogram for now
        // this.loadHistogram(buffer, colormapSubsectionOffset);
        buffer.position(saveOffset);
    }

    private void parseRpfColorOffsetRecord(java.nio.ByteBuffer buffer)
    {
        this.tableID                = NitfsUtil.getUShort(buffer);
        this.numOfColorRecords      = NitfsUtil.getUInt(buffer);
        this.colorElementLength     = NitfsUtil.getByteAsShort(buffer);
        this.histogramRecordLength  = NitfsUtil.getUShort(buffer);
        this.colorTableOffset       = NitfsUtil.getUInt(buffer);
        this.histogramTableOffset   = NitfsUtil.getUInt(buffer);
    }

    private void loadColorMaps(java.nio.ByteBuffer buffer, int colormapSubsectionOffset)
    {
        if (0 == this.numOfColorRecords)
            throw new NitfsRuntimeException("NitfsReader.InvalidNumberOfColorRecords");
        if (0 == this.colorElementLength)
            throw new NitfsRuntimeException("NitfsReader.InvalidLengthOfColorRecordElement");

        buffer.position((int) (colormapSubsectionOffset + this.colorTableOffset));
        int mapLength = (int)(this.numOfColorRecords * this.colorElementLength);
        this.colorMap = new byte[mapLength];
        buffer.get(this.colorMap, 0, mapLength);
    }

    private void loadHistogram(java.nio.ByteBuffer buffer, int colormapSubsectionOffset)
    {
        if (0 == this.numOfColorRecords)
            throw new NitfsRuntimeException("NitfsReader.InvalidNumberOfColorRecords");
        if (0 == this.histogramRecordLength)
            throw new NitfsRuntimeException("NitfsReader.InvalidLengthOfHistogramRecordElement");
        // skip the loading of the histogram table, just increment a position in the buffer
        buffer.position((int) (colormapSubsectionOffset + this.histogramTableOffset
            + (this.numOfColorRecords * this.histogramRecordLength)));
    }
}
