/*
Copyright (C) 2001, 2006 United States Government as represented by
the Administrator of the National Aeronautics and Space Administration.
All Rights Reserved.
*/
package gov.nasa.worldwind.formats.rpf;

import gov.nasa.worldwind.*;

import static java.util.logging.Level.*;

/**
 * @author dcollins
 * @version $Id: RpfProducer.java 1762 2007-05-07 19:43:55Z dcollins $
 */
public enum RpfProducer
{
    /* [Section 5.2.1, MIL-STD-2411-1] */
    Producer1('1', "AFACC", "Air Force Air Combat Command"),
    Producer2('2', "AFESC", "Air Force Electronic Systems Center"),
    Producer3('3', "NIMA", "National Imagery and Mapping Agency, Primary"),
    Producer4('4', "NIMA1", "NIMA, Alternate Site 1"),
    Producer5('5', "NIMA2", "NIMA, Alternate Site 2"),
    Producer6('6', "NIMA3", "NIMA, Alternate Site 3"),
    Producer7('7', "SOCAF", "Air Force Special Operations Command"),
    Producer8('8', "SOCOM", "United States Special Operations Command"),
    Producer9('9', "PACAF", "Pacific Air Forces"),
    ProducerA('A', "USAFE", "United States Air Force, Europe"),
    ProducerB('B', "Non-DoD (NonDD)", "US producer outside the Department of Defense"),
    ProducerC('C', "Non-US (NonUS)", "Non-US producer"),
    ProducerD('D', "NIMA", "DCHUM (DCHUM) NIMA produced Digital CHUM file"),
    ProducerE('E', "Non-NIMA DCHUM (DCHMD)", "DoD producer of Digital CHUM file otherthan NIMA "),
    ProducerF('F', "Non-US DCHUM (DCHMF)", "Non-US (foreign)producer of Digital CHUMfiles"),
    ProducerG('G', "Non-DoD DCHUM (DCHMG)", "US producer of Digital CHUM files outsideDoD"),
    ProducerH('H', "IMG2RPF", "Non-specified, Imagery formatted to RPF"),
//  Producer?('I'-'Z', "",                   "Reserved for future standardization"),
    ;

    public final Character id;
    public final String producerCode;
    public final String producer;

    private RpfProducer(Character id, String producerCode, String producer)
    {
        this.id = id;
        this.producer = producer;
        this.producerCode = producerCode;
    }

    private static RpfProducer[] enumConstantAlphabet = null;

    private static RpfProducer[] enumConstantAlphabet()
    {
        if (enumConstantAlphabet == null)
        {
            RpfProducer[] universe = RpfProducer.class.getEnumConstants();
            enumConstantAlphabet = new RpfProducer[36];
            for (RpfProducer producer : universe)
            {
                enumConstantAlphabet[indexFor(producer.id)] = producer;
            }
        }
        return enumConstantAlphabet;
    }

    private static int indexFor(Character id)
    {
        if (id >= '0' && id <= '9')
            return id - '0';
        else if (id >= 'A' && id <= 'Z')
            return 10 + id - 'A';
        return -1;
    }

    public static RpfProducer producerFor(Character id)
    {
        if (id == null)
        {
            String message = WorldWind.retrieveErrMsg("nullValue.CharacterIsNull");
            WorldWind.logger().log(FINE, message);
            throw new IllegalArgumentException(message);
        }
        RpfProducer zone;
        RpfProducer[] alphabet = enumConstantAlphabet();
        int index = indexFor(Character.toUpperCase(id));
        if (index < 0 || index >= alphabet.length || (zone = alphabet[index]) == null)
        {
            String message = WorldWind.retrieveErrMsg("generic.EnumNotFound") + id;
            WorldWind.logger().log(FINE, message);
            throw new EnumConstantNotPresentException(RpfZone.class, message);
        }
        return zone;
    }
}
