/*
Copyright (C) 2001, 2006 United States Government
as represented by the Administrator of the
National Aeronautics and Space Administration.
All Rights Reserved.
*/
package gov.nasa.worldwind.layers;

import gov.nasa.worldwind.*;

import javax.media.opengl.*;
import java.util.*;
import java.awt.*;

/**
 * @author tag
 * @version $Id: RenderableLayer.java 1435 2007-04-11 02:03:35Z tgaskins $
 */
public class RenderableLayer extends AbstractLayer
{
    private Collection<Renderable> renderables = new ArrayList<Renderable>();
    private final PickSupport pickSupport = new PickSupport();
    private final Layer delegateOwner;

    public RenderableLayer()
    {
        this.delegateOwner = null;
    }

    public RenderableLayer(Layer delegateOwner)
    {
        this.delegateOwner = delegateOwner;
    }

    public void setRenderables(Iterable<Renderable> shapeIterator)
    {
        this.renderables = new ArrayList<Renderable>();

        if (shapeIterator == null)
            return;

        for (Renderable renderable : shapeIterator)
        {
            this.renderables.add(renderable);
        }
    }

    public void addRenderable(Renderable renderable)
    {
        if (renderable == null)
        {
            String msg = WorldWind.retrieveErrMsg("nullValue.Shape");
            WorldWind.logger().log(java.util.logging.Level.FINE, msg);
            throw new IllegalArgumentException(msg);
        }

        this.renderables.add(renderable);
    }

    public void removeRenderable(Renderable renderable)
    {
        if (renderable == null)
        {
            String msg = WorldWind.retrieveErrMsg("nullValue.Shape");
            WorldWind.logger().log(java.util.logging.Level.FINE, msg);
            throw new IllegalArgumentException(msg);
        }

        this.renderables.remove(renderable);
    }

    public Collection<Renderable> getRenderables()
    {
        return this.renderables;
    }

    public void dispose()
    {
        for (Renderable renderable : this.renderables)
        {
            if (renderable instanceof Disposable)
                ((Disposable) renderable).dispose();
        }
    }

    @Override
    protected void doPick(DrawContext dc, java.awt.Point pickPoint)
    {
        this.pickSupport.clearPickList();
        this.pickSupport.beginPicking(dc);

        for (Renderable renderable : this.renderables)
        {
            float[] inColor = new float[4];
            dc.getGL().glGetFloatv(GL.GL_CURRENT_COLOR, inColor, 0);
            Color color = dc.getUniquePickColor();
            dc.getGL().glColor3ub((byte) color.getRed(), (byte) color.getGreen(), (byte) color.getBlue());

            renderable.render(dc);

            dc.getGL().glColor4fv(inColor, 0);

            if (renderable instanceof Locatable)
            {
                this.pickSupport.addPickableObject(color.getRGB(), renderable,
                    ((Locatable) renderable).getPosition(), false);
            }
            else
            {
                this.pickSupport.addPickableObject(color.getRGB(), renderable);
            }
        }

        this.pickSupport.resolvePick(dc, pickPoint, this.delegateOwner != null ? this.delegateOwner : this);
        this.pickSupport.endPicking(dc);
    }

    @Override
    protected void doRender(DrawContext dc)
    {
        for (Renderable renderable : this.renderables)
        {
            renderable.render(dc);
        }
    }

    public Layer getDelegateOwner()
    {
        return delegateOwner;
    }

    @Override
    public String toString()
    {
        return WorldWind.retrieveErrMsg("layers.RenderableLayer.Name");
    }
}
