/*
Copyright (C) 2001, 2006 United States Government
as represented by the Administrator of the
National Aeronautics and Space Administration.
All Rights Reserved.
*/
package gov.nasa.worldwind.layers;

import gov.nasa.worldwind.*;
import gov.nasa.worldwind.geom.*;

/**
 * @author tag
 * @version $Id: TrackLayer.java 1230 2007-03-16 14:47:35Z tgaskins $
 */
public class TrackLayer extends AbstractLayer
{
    private TrackRenderer trackRenderer = new TrackRenderer();
    private java.util.List<Track> tracks = new java.util.ArrayList<Track>();
    private Sector boundingSector;
    private IconRenderer iconRenderer = new IconRenderer();
    private UserFacingIcon icon;

    public TrackLayer(java.util.List<Track> tracks)
    {
        if (tracks == null)
        {
            String msg = WorldWind.retrieveErrMsg("nullValue.TracksIsNull");
            WorldWind.logger().log(java.util.logging.Level.FINE, msg);
            throw new IllegalArgumentException(msg);
        }

        this.tracks = tracks;
        this.boundingSector = Sector.boundingSector(this.iterator());
    }

    private TrackPointIteratorImpl iterator()
    {
        return new TrackPointIteratorImpl(this.tracks);
    }

    public void dispose()
    {
        this.trackRenderer.dispose();
    }

    public int getNumPoints()
    {
        return this.iterator().getNumPoints();
    }

    public double getMarkerPixels()
    {
        return this.trackRenderer.getMarkerPixels();
    }

    public void setMarkerPixels(double markerPixels)
    {
        this.trackRenderer.setMarkerPixels(markerPixels);
    }

    public double getMinMarkerSize()
    {
        return this.trackRenderer.getMinMarkerSize();
    }

    public void setMinMarkerSize(double minMarkerSize)
    {
        this.trackRenderer.setMinMarkerSize(minMarkerSize);
    }

    public double getMarkerElevation()
    {
        return this.trackRenderer.getMarkerElevation();
    }

    public void setMarkerElevation(double markerElevation)
    {
        this.trackRenderer.setMarkerElevation(markerElevation);
    }

    public Material getMaterial()
    {
        return this.trackRenderer.getMaterial();
    }

    public void setMaterial(Material material)
    {
        this.trackRenderer.setMaterial(material);
    }

    public String getIconFilePath()
    {
        return this.icon != null ? this.icon.getPath() : null;
    }

    public void setIconFilePath(String iconFilePath)
    {
        this.icon = iconFilePath != null ? new UserFacingIcon(iconFilePath, null) : null;
    }

    public int getLowerLimit()
    {
        return this.trackRenderer.getLowerLimit();
    }

    public void setLowerLimit(int lowerLimit)
    {
        this.trackRenderer.setLowerLimit(lowerLimit);
    }

    public int getUpperLimit()
    {
        return this.trackRenderer.getUpperLimit();
    }

    public void setUpperLimit(int upperLimit)
    {
        this.trackRenderer.setUpperLimit(upperLimit);
    }

    @Override
    protected void doPick(DrawContext dc, java.awt.Point pickPoint)
    {
        this.draw(dc, pickPoint);
    }

    protected void doRender(DrawContext dc)
    {
        this.draw(dc, null);
    }

    private void draw(DrawContext dc, java.awt.Point pickPoint)
    {
        TrackPointIterator trackPoints = this.iterator();
        if (!trackPoints.hasNext())
            return;
        
        if (dc.getVisibleSector() == null)
            return;

        SectorGeometryList geos = dc.getSurfaceGeometry();
        if (geos == null)
            return;

        if (!dc.getVisibleSector().intersects(this.boundingSector))
            return;

        Point iconPoint = this.trackRenderer.render(dc, trackPoints);

        if (iconPoint != null && this.icon != null)
        {
            if (dc.isPickingMode())
                this.iconRenderer.pick(dc, this.icon, iconPoint, pickPoint, this);
            else
                this.iconRenderer.render(dc, this.icon, iconPoint);
        }
    }

    @Override
    public String toString()
    {
        return gov.nasa.worldwind.WorldWind.retrieveErrMsg("layers.TrackLayer.Name");
    }
}