/*
Copyright (C) 2001, 2006 United States Government
as represented by the Administrator of the
National Aeronautics and Space Administration.
All Rights Reserved.
*/
package gov.nasa.worldwind;

import gov.nasa.worldwind.geom.*;
import gov.nasa.worldwind.geom.Point;

import javax.media.opengl.*;
import java.util.*;
import java.util.List;
import java.awt.*;

/**
 * @author tag
 * @version $Id: SectorGeometryList.java 1782 2007-05-08 06:27:54Z tgaskins $
 */
public class SectorGeometryList extends ArrayList<SectorGeometry>
{
    private PickSupport pickSupport = new PickSupport();

    public SectorGeometryList()
    {
    }

    public SectorGeometryList(SectorGeometryList list)
    {
        super(list);
    }

    public List<SectorGeometry> getIntersectingSectors(Sector sector)
    {
        ArrayList<SectorGeometry> list = null;

        for (SectorGeometry sg: this)
        {
            if (sg.getSector().intersects(sector))
            {
                if (list == null)
                    list = new ArrayList<SectorGeometry>();
                list.add(sg);
            }
            else
            {
//                System.out.println("no intersection");
            }
        }

        return list;
    }

    public void pick(DrawContext dc, java.awt.Point pickPoint)
    {
        this.pickSupport.clearPickList();
        this.pickSupport.beginPicking(dc);

        GL gl = dc.getGL();
        gl.glPushAttrib(GL.GL_LIGHTING_BIT | GL.GL_DEPTH_BUFFER_BIT | GL.GL_ENABLE_BIT | GL.GL_CURRENT_BIT);
        gl.glEnable(GL.GL_DEPTH_TEST);
        gl.glShadeModel(GL.GL_FLAT);
        gl.glDisable(GL.GL_CULL_FACE);

        try
        {
            // render each sector in unique color
            for (SectorGeometry sector : this)
            {
                Color color = dc.getUniquePickColor();
                dc.getGL().glColor3ub((byte) color.getRed(), (byte) color.getGreen(), (byte) color.getBlue());
                sector.render(dc);
                // lat/lon/elevation not used in this case
                this.pickSupport.addPickableObject(color.getRGB(), sector, Position.ZERO, true);
            }

            PickedObject pickedSector = this.pickSupport.getTopObject(dc, pickPoint, null);
            if (pickedSector == null || pickedSector.getObject() == null)
                return; // no sector picked

            SectorGeometry sector = (SectorGeometry) pickedSector.getObject();
            gl.glDepthFunc(GL.GL_LEQUAL);
            sector.pick(dc, pickPoint);
        }
        finally
        {
            gl.glPopAttrib();
            this.pickSupport.endPicking(dc);
            this.pickSupport.clearPickList();
        }
    }

    public Point getSurfacePoint(Position position)
    {
        return this.getSurfacePoint(position.getLatitude(), position.getLongitude(), position.getElevation());
    }

    public Point getSurfacePoint(Angle latitude, Angle longitude)
    {
        return this.getSurfacePoint(latitude, longitude, 0d);
    }

    public Point getSurfacePoint(Angle latitude, Angle longitude, double metersOffset)
    {
        if (latitude == null || longitude == null)
        {
            String msg = WorldWind.retrieveErrMsg("nullValue.LatLonIsNull");
            WorldWind.logger().log(java.util.logging.Level.FINE, msg);
            throw new IllegalArgumentException(msg);
        }

        for (SectorGeometry sg : this)
        {
            if (sg.getSector().contains(latitude, longitude))
            {
                Point point = sg.getSurfacePoint(latitude, longitude, metersOffset);
                if (point != null)
                    return point;
            }
        }

        return null;
    }
}
