/*
Copyright (C) 2001, 2006 United States Government
as represented by the Administrator of the
National Aeronautics and Space Administration.
All Rights Reserved.
*/
package gov.nasa.worldwind.formats.gpx;

import gov.nasa.worldwind.*;

/**
 * @author tag
 * @version $Id: ElementParser.java 513 2007-01-18 00:33:16Z ericdalgliesh $
 */
public class ElementParser
{
    protected final String elementName;
    protected ElementParser currentElement = null;
    protected String currentCharacters = null;

    /**
     * @param elementName the element's name, may not be null
     * @throws IllegalArgumentException if <code>elementName</code> is null
     */
    protected ElementParser(String elementName)
    {
        if (elementName == null)
        {
            String msg = WorldWind.retrieveErrMsg("nullValue.ElementNameIsNull");
            WorldWind.logger().log(java.util.logging.Level.FINE, msg);
            throw new IllegalArgumentException(msg);
        }

        this.elementName = elementName;
    }

    public String getElementName()
    {
        return this.elementName;
    }

    /**
     * Starts an element. No parameters may be null.
     *
     * @param uri
     * @param lname
     * @param qname
     * @param attributes
     * @throws org.xml.sax.SAXException
     * @throws IllegalArgumentException if any argument is null
     */
    public void startElement(String uri, String lname, String qname, org.xml.sax.Attributes attributes)
        throws org.xml.sax.SAXException
    {
        if (uri == null)
        {
            String msg = WorldWind.retrieveErrMsg("nullValue.URIIsNull");
            WorldWind.logger().log(java.util.logging.Level.FINE, msg);
            throw new IllegalArgumentException(msg);
        }
        if (lname == null)
        {
            String msg = WorldWind.retrieveErrMsg("nullValue.LNameIsNull");
            WorldWind.logger().log(java.util.logging.Level.FINE, msg);
            throw new IllegalArgumentException(msg);
        }
        if (qname == null)
        {
            String msg = WorldWind.retrieveErrMsg("nullValue.QNameIsNull");
            WorldWind.logger().log(java.util.logging.Level.FINE, msg);
            throw new IllegalArgumentException(msg);
        }
        if (attributes == null)
        {
            String msg = WorldWind.retrieveErrMsg("nullValue.org.xml.sax.AttributesIsNull");
            WorldWind.logger().log(java.util.logging.Level.FINE, msg);
            throw new IllegalArgumentException(msg);
        }

        if (this.currentElement != null)
            this.currentElement.startElement(uri, lname, qname, attributes);
        else
            this.doStartElement(uri, lname, qname, attributes);
    }

    /**
     * Finishes an element. No parameters may be null.
     *
     * @param uri
     * @param lname
     * @param qname
     * @throws org.xml.sax.SAXException
     * @throws IllegalArgumentException if any argument is null
     */
    public void endElement(String uri, String lname, String qname) throws org.xml.sax.SAXException
    {
        if (uri == null)
        {
            String msg = WorldWind.retrieveErrMsg("nullValue.URIIsNull");
            WorldWind.logger().log(java.util.logging.Level.FINE, msg);
            throw new IllegalArgumentException(msg);
        }
        if (lname == null)
        {
            String msg = WorldWind.retrieveErrMsg("nullValue.LNameIsNull");
            WorldWind.logger().log(java.util.logging.Level.FINE, msg);
            throw new IllegalArgumentException(msg);
        }
        if (qname == null)
        {
            String msg = WorldWind.retrieveErrMsg("nullValue.QNameIsNull");
            WorldWind.logger().log(java.util.logging.Level.FINE, msg);
            throw new IllegalArgumentException(msg);
        }
        if (this.currentElement != null)
        {
            this.currentElement.endElement(uri, lname, qname);
            if (lname.equalsIgnoreCase(this.currentElement.elementName))
                this.currentElement = null;
        }

        this.doEndElement(uri, lname, qname);

        this.currentCharacters = null;
    }

    protected void doStartElement(String uri, String lname, String qname, org.xml.sax.Attributes attributes)
        throws org.xml.sax.SAXException
    {
    }

    protected void doEndElement(String uri, String lname, String qname) throws org.xml.sax.SAXException
    {
    }

    /**
     * @param data
     * @param start
     * @param length
     * @throws IllegalArgumentException if <code>data</code> has length less than 1
     */
    public void characters(char[] data, int start, int length)
    {
        if (data == null)
        {
            String msg = WorldWind.retrieveErrMsg("nullValue.ArrayIsNull");
            WorldWind.logger().log(java.util.logging.Level.FINE, msg);
            throw new IllegalArgumentException(msg);
        }
        if (data.length < 1)
        {
            String msg = WorldWind.retrieveErrMsg("generic.arrayInvalidLength") + WorldWind.retrieveErrMsg(
                "punctuation.space") + data.length;
            WorldWind.logger().log(java.util.logging.Level.FINE, msg);
            throw new IllegalArgumentException(msg);
        }
        if (start < 0 || length < 0)
        {
            String msg = WorldWind.retrieveErrMsg("generic.indexOutOfRange");
            WorldWind.logger().log(java.util.logging.Level.FINE, msg);
            throw new IllegalArgumentException(msg);
        }
        if (start + length > data.length)
        {
            String msg = WorldWind.retrieveErrMsg("generic.indexOutOfRange") + WorldWind.retrieveErrMsg(
                "punctuation.space") + String.valueOf(start + length);
            WorldWind.logger().log(java.util.logging.Level.FINE, msg);
            throw new IllegalArgumentException(msg);
        }

        if (this.currentElement != null)
            this.currentElement.characters(data, start, length);
        else if (this.currentCharacters != null)
            this.currentCharacters += new String(data, start, length);
        else
            this.currentCharacters = new String(data, start, length);
    }
}
